# coding=utf-8
from __future__ import unicode_literals

import base64
import logging
import random
from collections import defaultdict
from datetime import datetime, timedelta
from time import time, sleep

import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2

from sandbox.projects.avia.lib.yt_helpers import tables_for_daterange
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

log = logging.getLogger(__name__)


class RedirLogAnalyzer(object):
    def __init__(self, partner, redir_log, yt_client, time_window=90):
        self.partner = partner
        self.redir_log = redir_log
        self.time_window = time_window
        self.yt_client = yt_client

    def _filter_out_other_partners_mapper(self, input_row):
        if input_row['partnerCode'] == self.partner:
            yield {'fromId': input_row['fromId'], 'toId': input_row['toId']}

    @staticmethod
    def _count_distinct_directions_reducer(key, input_row_iterator):
        from_id = key['fromId']
        distinct_to = defaultdict(int)
        for input_row in input_row_iterator:
            distinct_to[input_row['toId']] += 1

        for to_id in distinct_to:
            yield {'from_id': from_id, 'to_id': to_id, 'count': distinct_to[to_id]}

    def _get_input_table_names(self):
        today = datetime.now().date()
        return tables_for_daterange(
            self.yt_client,
            self.redir_log,
            today - timedelta(days=self.time_window),
            today,
        )

    def get_popular_directions(self):
        sorted_tmp_table = self.yt_client.create_temp_table()
        input_tables = self._get_input_table_names()

        self.yt_client.run_map_reduce(
            self._filter_out_other_partners_mapper,
            self._count_distinct_directions_reducer,
            source_table=input_tables,
            destination_table=sorted_tmp_table,
            reduce_by=["fromId"])

        popular_directions = list(self.yt_client.read_table(sorted_tmp_table))
        popular_directions.sort(key=lambda x: x['count'], reverse=True)
        for direction in popular_directions:
            log.info(direction)
        return popular_directions


class AVIA_REVISE_SCREENSHOT(sdk2.Resource):
    pass


class QueryInitializer(object):
    def __init__(self, driver, search_page_url):
        self.driver = driver
        self.driver.implicitly_wait(30)
        self.driver.set_window_size(2560, 1440)
        self.search_page_url = search_page_url

    def query(self, query, partner, task_instance):
        url = self.build_query_string(query, partner)
        log.info('Querying: %s', url)
        try:
            self.driver.get(url)
            sleep(40)
            top_result = self.driver.find_element_by_xpath(
                '(//span[@class="YTButton-Text" and contains(text(), "Купить")])[1]')
            self.driver.execute_script("arguments[0].click();", top_result)
            cheapest_company_selector = '//a[@qa="cheapest-company-avia-order-offers-offer-button"]'
            cheapest_company_button = self.driver.find_element_by_xpath(cheapest_company_selector)
            self.driver.execute_script("arguments[0].click();", cheapest_company_button)
        except Exception:
            self.save_screenshot(task_instance, '{}_exception.png'.format(partner))
            log.exception('Failed query url %s', url)
            return False
        self.save_screenshot(task_instance, '{}_avia_page.png'.format(partner))
        window_handles = self.driver.window_handles
        self.driver.switch_to.window(window_handles[-1])
        sleep(5)
        self.save_screenshot(task_instance, '{}_partner_page.png'.format(partner))

        return True

    def save_screenshot(self, task_instance, name=None):
        resource = sdk2.ResourceData(AVIA_REVISE_SCREENSHOT(
            task_instance, "Screenshots", "screenshots"
        ))
        resource.path.mkdir(0o755, parents=True, exist_ok=True)
        if not name:
            name = 'screenshot_{}.png'.format(time())
        else:
            name = '{}_{}.png'.format(name, time())
        resource.path.joinpath(name).write_bytes(self.driver.get_screenshot_as_png())

    def build_query_string(self, query, partner):
        return '{}?klass=economy&fromId={}&toId={}&when={}&return_date={}&adult_seats={}' \
               '&children_seats={}&infant_seats={}#pt={}' \
            .format(self.search_page_url, query['from_id'], query['to_id'], query['date_forward'],
                    query.get('date_backward', ''),
                    query.get('adults', 1), query.get('children', 0), query.get('infants', 0), partner)


class AviaTriggerRevise(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    _yt_client = None

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        environments = [sdk_environments.PipEnvironment('yandex-yt', version='0.10.8'),
                        sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
                        sdk_environments.PipEnvironment('raven'),
                        sdk_environments.PipEnvironment('selenium'),
                        ]

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Yt settings') as yt_block:
            yt_cluster = sdk2.parameters.String('Yt cluster', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('Yt token', required=True, default='sec-01dfxmszhq27tk66hm107d0ycd')
            redirects_folder = sdk2.parameters.String('Redirects folder', required=True,
                                                      default='//home/avia/logs/avia-json-redir-log')

        with sdk2.parameters.Group('Task settings') as task_block:
            partner = sdk2.parameters.String('Partner code', required=True)
            search_page_url = sdk2.parameters.String('Search page url', required=True,
                                                     default='https://travel-test.yandex.ru/avia/search/result/')
            required_redirects = sdk2.parameters.Integer('Number of successful redirects', required=True, default=40)
            maximum_tries = sdk2.parameters.Integer('Total number of tries', required=True, default=100)

        with sdk2.parameters.Group('Selenium settings') as selenium_block:
            selenium_quota = sdk2.parameters.String('Selenium quota', required=True, default='avia')
            selenium_password = sdk2.parameters.YavSecret('Selenium password', required=True,
                                                          default='sec-01dv8b2n6z8nqd00ad0bdy47x2')
            ignore_ssl_errors = sdk2.parameters.Bool('Ignore ssl certificate errors', default=False)
        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        super(AviaTriggerRevise, self).on_execute()

        redir_log_analyzer = RedirLogAnalyzer(self.Parameters.partner,
                                              self.Parameters.redirects_folder,
                                              self._get_yt_client(),
                                              10)

        result = redir_log_analyzer.get_popular_directions()

        total_queries, successful_queries = 0, 0
        query_initializer = QueryInitializer(self._get_selenium_driver(), self.Parameters.search_page_url)
        for direction in result:
            if successful_queries >= self.Parameters.required_redirects:
                break
            if total_queries >= self.Parameters.maximum_tries:
                raise SandboxTaskFailureError('Allowed tries exhausted. Exiting')
            for query in self.build_queries_for_direction(direction):
                successful_queries += query_initializer.query(query, self.Parameters.partner, self)
                total_queries += 1
        if successful_queries < self.Parameters.required_redirects:
            log.info('Run out of popular directions. Made %s successful redirects out of %s required',
                     successful_queries, self.Parameters.required_redirects)
        log.info('Done. Made %s queries, %s of them successfully', total_queries, successful_queries)

    @staticmethod
    def build_queries_for_direction(direction):
        query = {'from_id': direction['from_id'],
                 'to_id': direction['to_id']}
        date_forward = datetime.today() + timedelta(days=random.randint(2, 7))
        date_backward = (date_forward + timedelta(days=random.randint(0, 14))).strftime('%Y-%m-%d')
        query['date_forward'] = date_forward.strftime('%Y-%m-%d')

        yield dict(query)
        yield dict(query, date_backward=date_backward)
        yield dict(query, adults=2, children=1, infants=1)

    def _get_yt_client(self):
        if self._yt_client is None:
            from projects.avia.lib.yt_helpers import YtClientFactory
            self._yt_client = YtClientFactory.create(
                proxy=self.Parameters.yt_cluster,
                token=self.Parameters.yt_token.data()['token']
            )
        return self._yt_client

    def _get_selenium_driver(self):
        from selenium import webdriver
        from selenium.webdriver.remote.remote_connection import RemoteConnection
        selenium_grid_url = 'http://{user}:{password}@sg.yandex-team.ru:4444/wd/hub'.format(
            user=self.Parameters.selenium_quota,
            password=self.Parameters.selenium_password.data()['password'],
        )
        driver = webdriver.Remote(
            command_executor=RemoteConnection(selenium_grid_url, resolve_ip=False),
            desired_capabilities={
                'platform': 'ANY',
                'browserName': 'chrome',
                'version': '80.0',
                'acceptSslCerts': self.Parameters.ignore_ssl_errors
            }
        )

        return driver
