# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import date


from requests import Session
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry
from six.moves.urllib.parse import urljoin

from sandbox import sdk2
from sandbox.projects.common import binary_task

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.pg_helpers import get_psycopg2_conn_string

log = logging.getLogger(__name__)

BETTER_PRICE_SUBSCRIPTIONS_PG_TABLE_NAME = 'better_price_subscriptions'


class AviaUpdateYeahConfig(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 4096  # 4096 Megs
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Binary task release parameters') as binary_task_release_parameters:
            binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YEAH settings') as yeah_settings:
            yeah_url = sdk2.parameters.String('YEAH url', required=True, default='http://yeah.testing.avia.yandex.net')
            better_price_subscriptions_config_key = sdk2.parameters.String(
                'Better price subscriptions config key',
                required=True,
                default='better_price_subscriptions',
            )

        with sdk2.parameters.Group('Notifier Postgresql settings') as notifier_pg_settings:
            notifier_pg_host = sdk2.parameters.String('Postgresql host name', required=True,
                                                      default='c-mdb9sssbmtcje8gtvlrc.rw.db.yandex.net')
            notifier_pg_port = sdk2.parameters.String('Postgresql port', required=True, default=6432)
            notifier_pg_user = sdk2.parameters.String('Postgresql user', default='avia', required=True)
            notifier_pg_dbname = sdk2.parameters.String('Postgresql dbname', default='notifier', required=True)
            notifier_pg_password_vault = sdk2.parameters.YavSecret('Postgresql password', required=True,
                                                                   default='sec-01e84cae2rjpe3pvwjrt9c0phb')

    def on_execute(self):
        log.info('Start')
        self._process_better_price_subscriptions()
        log.info('Finish')

    def _process_better_price_subscriptions(self):
        import psycopg2
        from psycopg2.extras import LoggingConnection

        pg_connection_string = get_psycopg2_conn_string(
            host=self.Parameters.notifier_pg_host,
            port=self.Parameters.notifier_pg_port,
            db_name=self.Parameters.notifier_pg_dbname,
            user=self.Parameters.notifier_pg_user,
            password=self.Parameters.notifier_pg_password_vault.data()['password'],
        )

        log.info('Setting up connection to notifier\'s postgres')
        with psycopg2.connect(pg_connection_string, connection_factory=LoggingConnection) as connection:
            connection.initialize(log)

            log.info('Getting better price subscriptions from postgres')
            better_price_subscriptions = self._get_better_price_subscriptions(connection)
            log.info('Building config for subscriptions')
            config = self._build_heater_config_for_better_price_subscriptions(better_price_subscriptions)
            log.info('Uploading config to YEAH')
            self._update_heater_config(config, self.Parameters.better_price_subscriptions_config_key, 'many_heater_configs')

    def _get_better_price_subscriptions(self, pg_connection):
        today = date.today().strftime('%Y-%m-%d')
        with pg_connection.cursor() as cursor:
            cursor.execute('SELECT * FROM {table} WHERE date_forward > {today}'.format(
                table=BETTER_PRICE_SUBSCRIPTIONS_PG_TABLE_NAME,
                today=today,
            ))
            return cursor.fetchall()

    def _build_heater_config_for_better_price_subscriptions(self, better_price_subscriptions):
        return [
            {
                'code_from': s['from_point_key'],
                'code_to': s['to_point_key'],
                'forward_date': s['date_forward'],
                'backward_date': s['date_backward'],
                'national_version': s.get('national_version', 'ru'),
                'lang': s.get('lang'),
                'adults_count': s['adults'],
                'children_count': s['children'],
                'infants_count': s['infants']

            }
            for s in better_price_subscriptions
        ]

    def _update_heater_config(self, config_id, config, config_format):
        session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(500, 502, 503, 504),
                method_whitelist=('POST',),
            ),
        )
        session.mount('http://', adapter)
        session.mount('https://', adapter)

        query_params = '?format={}'.format(config_format)
        request_url = urljoin(self.Parameters.yeah_url, 'heater_config/custom/')
        request_url = urljoin(request_url, config_id) + query_params

        session.post(request_url, json=config)
