# coding: utf-8
import datetime
import logging

import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sdk2 as sdk2

from sandbox.projects.avia.base import AviaBaseTask


class AviaAbstractVerifyLogTask(AviaBaseTask):
    _yql_client = None

    @property
    def _invalid_log_records_query(self):
        # type: () -> str
        raise NotImplementedError()

    @property
    def _logs_path(self):
        return self.Parameters.logs_path

    @property
    def _error_log_records_limit(self):
        return int(self.Parameters.error_log_records_limit)

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yt_cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)
            yql_token_vault_owner = sdk2.parameters.String('YQL token vault owner', required=True, default='AVIA')
            yql_token_vault_name = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')

        with sdk2.parameters.Group('Logs settings') as logs_settings:
            logs_path = sdk2.parameters.String('Logs path', required=True, default='//home/avia/logs')

        with sdk2.parameters.Group('Error log settings') as error_log_settings:
            error_log_records_limit = sdk2.parameters.Integer(
                'Log only first N invalid records',
                required=True,
                default=10,
            )

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            sdk_environments.PipEnvironment('yql'),
            sdk_environments.PipEnvironment('yandex-yt', version='0.10.8'),
            sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        )

    @property
    def yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                db=str(self.Parameters.yt_cluster),
                token=sdk2.Vault.data(self.Parameters.yql_token_vault_owner, self.Parameters.yql_token_vault_name),
            )

        return self._yql_client

    def _get_query_parameters(self, table_path, date_time):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        params = {
            '$table_path': ValueBuilder.make_string(table_path),
        }

        if '/30min' in self._logs_path:
            params.update({
                '$table_start_date': ValueBuilder.make_string(date_time.strftime('%Y-%m-%dT00:00:00')),
                '$table_finish_date': ValueBuilder.make_string(date_time.strftime('%Y-%m-%dT23:59:59')),
            })
        else:
            params.update({
                '$table_start_date': ValueBuilder.make_string(date_time.strftime('%Y-%m-%d')),
                '$table_finish_date': ValueBuilder.make_string(date_time.strftime('%Y-%m-%d')),
            })

        return ValueBuilder.build_json_map(params)

    def _verify_log(self, date_time):
        request = self.yql_client.query(self._invalid_log_records_query, syntax_version=1)
        request.run(self._get_query_parameters(self._logs_path, date_time))
        for table in request.get_results():
            table.fetch_full_data()
            rows_count = len(table.rows)
            if rows_count > 0:
                logging.error(
                    'Log "%s" at %s verification failed. %s records are invalid.',
                    self._logs_path,
                    date_time.strftime('%Y-%m-%d'),
                    rows_count,
                )
                logging.error(
                    'Listing %s out of %s invalid records:',
                    min(rows_count, self._error_log_records_limit),
                    rows_count,
                )
                for i, row in enumerate(table.rows[:self._error_log_records_limit]):
                    logging.error('Record %s out of %s', i + 1, rows_count)
                    for column_name, column_value in zip(table.column_names, row):
                        logging.error('%s: %s', column_name, column_value)
                return False
            else:
                logging.info('Log %s verification was successful.', self._logs_path)
                return True

    def on_execute(self):
        today = datetime.date.today()
        if not self._verify_log(today):
            raise RuntimeError('Logs "{}" verification failed for {}'.format(
                self._logs_path, today.strftime('%Y-%m-%d'),
            ))

        yesterday = today - datetime.timedelta(days=1)
        if not self._verify_log(yesterday):
            raise RuntimeError('Logs "{}" verification failed for {}'.format(
                self._logs_path, yesterday.strftime('%Y-%m-%d'),
            ))

        logging.info('%s verification succeeded', self._logs_path)
