# -*- encoding: utf-8 -*-
import logging
import os

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.common.types.misc as ctm

ARCADIA_URL = 'arcadia-arc:/#trunk'


class AviaZeroDiffReleaseTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Parameters') as executable_params:
            stage = sdk2.parameters.String(
                'Stage',
                required=True,
            )
            deploy_unit = sdk2.parameters.String(
                'Deploy unit',
                required=True,
            )
            with sdk2.parameters.RadioGroup('Target type') as target_type:
                target_type.values['box'] = target_type.Value('box', default=True)
                target_type.values['workload'] = target_type.Value('workload')

            target = sdk2.parameters.String(
                'Target',
                description="workload or box",
                required=True,
            )
            cause = sdk2.parameters.String(
                'Release cause',
                default='sandbox_night',
                required=True,
            )
        with sdk2.parameters.Group('Secrets') as secrets_settings:
            dctl_yp_token = sdk2.parameters.YavSecret(
                'Yav secret with DCTL token',
                required=True,
                default='sec-01exyrpzw0wmecz2aeph888fpj',
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)
        super(AviaZeroDiffReleaseTask, self).on_prepare()

    def on_execute(self):
        logging.info('Execution started')
        super(AviaZeroDiffReleaseTask, self).on_execute()

        logging.info('Creating env variables')
        os.environ['DCTL_YP_TOKEN'] = self.get_secret_value(self.Parameters.dctl_yp_token, 'token')

        with arcadia_sdk.mount_arc_path(ARCADIA_URL, use_arc_instead_of_aapi=True) as arcadia_src_dir:
            self._release(dctl=[
                arcadia_sdk._python(),
                arcadia_sdk._get_ya_tool(arcadia_src_dir),
                'tool', 'dctl',
            ])

    def _release(self, dctl):
        from travel.avia.devops.zero_diff_releaser.lib.releaser import Releaser
        try:
            releaser = Releaser(**{
                'stage': self.Parameters.stage,
                'deploy_unit': self.Parameters.deploy_unit,
                self.Parameters.target_type: self.Parameters.target,
                'cause': self.Parameters.cause,
                'dctl': dctl,
            })
            self._logger.info('Start releasing')
            releaser.run()
            self._logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError('Unable to release stage {}. {}'.format(self.Parameters.stage, e))
