import collections
import logging
import datetime
import os.path

import requests

from sandbox import sdk2
from sandbox import common
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2 import ssh


AWACS_OAUTH_TOKEN = 'awacs_nanny_robot_oauth_token'
SVN_URL = 'arcadia:/arc/trunk/arcadia/infra/awacs/mirror'
SVN_USER = 'robot-awacs'


class BackupAwacsConfigs(sdk2.Task):
    """Backup Awacs Configs and commit them to Arcadia"""

    class Parameters(sdk2.Parameters):
        notifications = [
            sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.Group.BREAK],
                recipients=['awacs-alerts'],
                transport=ctn.Transport.TELEGRAM
            )
        ]

    releasers = ['pirogov', 'romanovich', 'disafonov', 'ferenets']

    def on_execute(self):
        awacs_token = self.get_token(AWACS_OAUTH_TOKEN)
        root_dir = os.getcwd()
        logging.info('root dir is {}'.format(root_dir))
        self.set_info('root dir is {}'.format(root_dir))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("awacsctl")) as pl:
            # get latest awacsctl module
            retcode = sp.Popen(
                "curl -vOJ https://proxy.sandbox.yandex-team.ru/last/AWACS_CTL_LINUX && chmod +x awacsctl2", shell=True,
                stdout=pl.stdout, stderr=sp.STDOUT).wait()
            if retcode != 0:
                raise common.errors.TaskFailure("Error loading awacsctl2")
            # test awacsctl
            logging.info('Home dir is {}'.format(os.environ['HOME']))
            with open('{}/.awacsctl.cfg'.format(os.environ['HOME']), 'w') as cfg_file:
                cfg_file.write("awacs_url: 'https://awacs.yandex-team.ru/'\nawacs_token: {}".format(awacs_token))
            retcode = sp.Popen("{}/awacsctl2 --version".format(root_dir), shell=True, stdout=pl.stdout,
                               stderr=sp.STDOUT).wait()
            if retcode != 0:
                raise common.errors.TaskFailure("Error getting awacsctl2 version")

            opts = Arcadia.SvnOptions()
            opts.force = True

            templates_path = Arcadia.get_arcadia_src_dir(SVN_URL)
            logging.info('templates dir is {}'.format(templates_path))
            namespaces = self.get_awacs_namespaces(awacs_token)
            os.chdir(templates_path)
            for ns in namespaces:
                # temporary skip yappy.yandex.ru
                # if ns == 'yappy.yandex.ru':
                #    continue
                # temporary skip (https://st.yandex-team.ru/AWACS-1291)
                if ns == 'uac.test.yandex.ru':
                    continue

                # skip namespaces with . in the end - Arcadia does not allow such dirs
                retcode = 0
                if ns.endswith('.'):
                    logging.warn('Wrong name format - {}'.format(ns))
                    continue
                if not os.path.isdir(ns):
                    logging.info('Clone awacs-ns: {}'.format(ns))
                    retcode = self.awacsctl_clone(root_dir, pl, ns)
                else:
                    logging.info('Pull awacs-ns: {}'.format(ns))
                    retcode = self.awacsctl_pull(root_dir, pl, ns)
                logging.debug('result - {}'.format(retcode))
                if retcode != 0:
                    self.set_info("Error getting namespace {}".format(ns))
                    namespaces.remove(ns)
                    # raise common.errors.TaskFailure("Error getting namespace {}".format(ns))

            # remove CRLFs
            logging.info('removing CRLFs')
            self.post_process_namespaces(templates_path)

            # delete obsolete namespaces, backends, upstreams
            logging.info('delete obsolete namespaces')
            self.cleanup_namespaces(namespaces, templates_path)

            logging.info('lowercasing files')
            self.lowercase_files(templates_path)

            opts = Arcadia.SvnOptions()
            opts.force = True
            opts.depth = 'infinity'
            logging.info('adding all the changes to svn')
            result = Arcadia.svn('add', opts=opts, path=templates_path)
            if result.returncode != 0:
                logging.warn(result)

            with ssh.Key(self, 'NANNY', 'awacs_ssh_key'):
                logging.info('commit changes to svn')
                Arcadia.commit(templates_path,
                               'daily backup, {:%d.%m.%Y %H:%M} SKIP_CHECK'.format(datetime.datetime.now()), SVN_USER)

    def on_finish(self, prev_status, status):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("curl")) as pl:
            sp.Popen("rm ./awacsctl2", shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()

    # clone namespaces (for new namespaces)
    def awacsctl_clone(self, root_dir, pl, ns):
        return sp.Popen("{}/awacsctl2 clone {}".format(root_dir, ns), shell=True, stdout=pl.stdout,
                        stderr=sp.STDOUT).wait()

    # pull namespace (for existing namespaces)
    def awacsctl_pull(self, root_dir, pl, ns):
        return sp.Popen("{}/awacsctl2 pull {} --override".format(root_dir, ns), shell=True, stdout=pl.stdout,
                        stderr=sp.STDOUT).wait()

    # remove CRLFs from configs (CRLFs are not allowed in Arcadia)
    def post_process_namespaces(self, ns_dir=None):
        for (dirpath, dirnames, filenames) in os.walk(ns_dir if ns_dir is not None else os.getcwd()):
            for f in filenames:
                if '.svn' in dirpath:
                    # print dirpath, f
                    continue
                with open('{}/{}'.format(dirpath, f), 'r+') as cfg_file:
                    data = cfg_file.read()
                    if data.find('\r') != -1:
                        logging.info('remove clrf from {}/{}'.format(dirpath, f))
                        data = data.replace('\r', '')
                        cfg_file.seek(0)
                        cfg_file.write(data)
                        cfg_file.truncate()

    # lowercase all files
    def lowercase_files(self, ns_dir=None):
        for (dirpath, dirnames, filenames) in os.walk(ns_dir if ns_dir is not None else os.getcwd()):
            d = collections.defaultdict(list)
            for f in filenames:
                d[f.lower()].append(f)
            for k, v in d.items():
                if len(v) > 1:
                    logging.info('conflicting files in {}: {}'.format(v, dirpath))

            for f in filenames:
                if '.svn' in dirpath:
                    # print dirpath, f
                    continue

                old = os.path.join(dirpath, f)
                new = os.path.join(dirpath, d[f.lower()][0])
                if old != new:
                    logging.info('renaming {} to {}'.format(old, new))
                    os.rename(old, new)

    # delete deprecated namespaces, backends, upstreams
    def cleanup_namespaces(self, namespaces, ns_dir=None):
        root_dir = True
        for (dirpath, dirnames, filenames) in os.walk(ns_dir if ns_dir is not None else os.getcwd()):
            if root_dir:
                for dir_name in dirnames:
                    # print dirpath, f
                    if dir_name not in namespaces and dir_name != '.svn':
                        logging.info('deleted namespace {}. Need to cleanup'.format(dir_name))
                        result = Arcadia.svn('del', path=dir_name)
                        if result.returncode != 0:
                            logging.warn(result)
                root_dir = False
            else:
                for f in filenames:
                    if f.endswith('.DELETED'):
                        logging.info('need to delete {}/{} from arcadia'.format(dirpath, f.replace('.DELETED', '')))
                        result = Arcadia.svn('del', path='{}/{}'.format(dirpath, f.replace('.DELETED', '')))
                        if result.returncode != 0:
                            logging.warn(result)

    # get list of awacs namespaces
    def get_awacs_namespaces(self, token):
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(token)
        session.headers['Content-Type'] = 'application/json'
        namespaces = list()

        result = session.post(
            'https://awacs.yandex-team.ru/api/ListNamespaceSummaries/',
            json={}
        ).json()
        logging.debug(result)
        for item in result['summaries']:
            namespaces.append(item['id'])

        return namespaces

    def get_token(self, secret_name):
        try:
            return sdk2.Vault.data(secret_name)
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Token not found!")
