import logging
import os.path
import shlex
import subprocess
import datetime
import tarfile
import contextlib

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk import environments


class AwacsStorageBackup(sdk2.Resource):
    auto_backup = True


class BackupAwacsStorage(sdk2.Task):
    class Parameters(sdk2.Parameters):
        zk_addr = sdk2.parameters.String(
            'Zookeeper address (e.g. myt0-1219.search.yandex.net:2184)',
            required=True
        )

        awacs_root_zk_node = sdk2.parameters.String(
            'Which awacs zk root node to backup',
            required=True,
            choices=[
                ('production_awacs', 'production_awacs'),
                ('prestable_awacs', 'prestable_awacs'),
            ]
        )

        nanny_root_zk_node = sdk2.parameters.String(
            'Which nanny zk root node to backup',
            required=True,
            choices=[
                ('nanny', 'nanny'),
                ('nanny_test', 'nanny_test'),
            ]
        )

        notifications = [
            sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.Group.BREAK],
                recipients=['awacs-alerts', 'nanny-monitorings'],
                transport=ctn.Transport.TELEGRAM
            )
        ]

    MIN_SUCCESSFUL_BACKUP_SIZE = 100 * 1024  # https://st.yandex-team.ru/AWACS-947

    # https://nanny.yandex-team.ru/ui/#/services/catalog/production_alemate_zookeeper/
    ZK_SHELL_CMD_TEMPLATE = ('zk-shell {zk_addr} --run-once "mirror {zk_path} json://{zk_node}.json/ '
                             '{async_} {verbose} {skip_prompt}"')
    ZK_NODES_TO_BACKUP = [
        'balancers_2',
        'balancer_states_2',
        'balancer_aspects_sets',
        'namespaces',
        'namespace_aspects_sets',
        'components',
        'knobs',
        'certs',
        'cert_renewals',
        'l3_balancers',
        'l3_balancer_states',
        'dns_records',
        'dns_record_states',
        'name_servers',
        'backends',
        'endpoint_sets',
        'upstreams',
        'domains',
        'domain_operations',
        'balancer_operations',
    ]

    NANNY_ZK_NODES_TO_BACKUP = [
        'services',
        'replication_policies',
        'service_notifications',
    ]

    def get_zk_shell_cmd(self, zk_addr, root_zk_node, zk_node, async_=False, verbose=False, skip_prompt=False):
        return self.ZK_SHELL_CMD_TEMPLATE.format(
            zk_addr=zk_addr,
            zk_path='/{}/{}'.format(root_zk_node, zk_node),
            zk_node=zk_node,
            async_=str(async_).lower(),
            verbose=str(verbose).lower(),
            skip_prompt=str(skip_prompt).lower(),
        )

    def on_execute(self):
        self.backup_dir = os.path.abspath('backup')
        make_folder(self.backup_dir)
        zk_addr = self.Parameters.zk_addr
        awacs_root_zk_node = self.Parameters.awacs_root_zk_node
        assert awacs_root_zk_node in ('production_awacs', 'prestable_awacs')
        nanny_root_zk_node = self.Parameters.nanny_root_zk_node
        assert nanny_root_zk_node in ('nanny', 'nanny_test')

        with environments.VirtualEnvironment() as venv:
            venv.pip('pip==9.0.3')
            self._run('pip install zk_shell --index-url https://pypi.yandex-team.ru/simple/')
            commands = [
                self.get_zk_shell_cmd(zk_addr=zk_addr,
                                      root_zk_node=awacs_root_zk_node,
                                      zk_node=zk_node,
                                      verbose=True,
                                      skip_prompt=True)
                for zk_node in self.ZK_NODES_TO_BACKUP
            ]

            commands.extend([
                self.get_zk_shell_cmd(zk_addr=zk_addr,
                                      root_zk_node=nanny_root_zk_node,
                                      zk_node=zk_node,
                                      verbose=True,
                                      skip_prompt=True)
                for zk_node in self.NANNY_ZK_NODES_TO_BACKUP
            ])

            for cmd in commands:
                logging.info('running "%s"', cmd)
                self._run(cmd, cwd=self.backup_dir)

            self._make_resource()

    def _make_resource(self):
        logging.info('creating tgz file')
        today_date = datetime.date.today()
        description = 'awacs and nanny storage backup {}'.format(today_date.strftime('%d/%m/%Y'))
        resource = sdk2.ResourceData(AwacsStorageBackup(
            task=self,
            description=description,
            path='backup.tar.gz'
        ))

        with contextlib.closing(tarfile.open(str(resource.path), 'w:gz')) as tar:
            for entry in os.listdir(self.backup_dir):
                tar.add(os.path.join(self.backup_dir, entry), entry)

        # zk-shell sometimes exit with 0 even if failed to connect to zk and did not mirror anything,
        # let's introduce an additional sanity check
        backup_size = os.path.getsize(str(resource.path))
        if backup_size < self.MIN_SUCCESSFUL_BACKUP_SIZE:
            raise Exception('something wen wrong, backup size is too small: {}'.format(backup_size))

        resource.ready()

    def _run(self, command, cwd=None):
        command = shlex.split(command)
        popen = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, cwd=cwd)
        stdout, stderr = popen.communicate()
        if popen.returncode:
            logging.error('stdout: %s', stdout)
            logging.error('stderr: %s', stderr)
            raise Exception('{} non zero return code: {}'.format(command, popen.returncode))
