import logging
import re
from requests.exceptions import Timeout, HTTPError
import time

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.conductor import Api, OauthAuth
from sandbox.projects.common.decorators import retries


VAULT_OWNER = "BILLING-CI"
VAULT_NAME = "conductor_token"

TICKET_ID_REGEX = re.compile(r"Ticket (\d+)", re.MULTILINE)


class ConductorStatus(object):
    NEW = "new"
    DONE = "done"
    OBSOLETE = "obsolete"
    FAILED = "failed"
    IN_WORK = "in_work"


class ConductorStatusGroup(object):
    DONE = [ConductorStatus.DONE]
    FAILED = [ConductorStatus.FAILED, ConductorStatus.OBSOLETE]
    PROCESSING = [ConductorStatus.NEW, ConductorStatus.IN_WORK]


class CreateConductorTicket(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        packages = sdk2.parameters.Dict(
            "Ticket packages",
            required=True,
            description="{package: version} dict format",
        )
        branch = sdk2.parameters.String(
            "Ticket branch",
            required=True,
        )
        filters = sdk2.parameters.List(
            "Ticket filters",
            required=True,
            description="<project>/<workflow>/<deploy_group> or <project>/<workflow> format",
        )
        with sdk2.parameters.Group("Waiting parameters") as scraper_waiting_group:
            active_waiting_duration = sdk2.parameters.Integer(
                "Waiting time with running task (in seconds)",
                default=60 * 20,  # 20 minutes
            )
            passive_waiting_tries = sdk2.parameters.Integer(
                "How many times task should go into WaitTime status",
                default=10,
            )
            passive_waiting_duration = sdk2.parameters.Integer(
                "Duration of WaitTime status",
                default=60 * 60,  # 1 hour
            )

        with sdk2.parameters.Output:
            ticked_id = sdk2.parameters.String(
                "Id of created ticket", required=True
            )
            ticket_url = sdk2.parameters.String(
                "Url of created ticket", required=True
            )

    def _check_status(self, ticket_id, ticket_status_func):
        status = ticket_status_func(ticket_id)
        logging.info("Get conductor status: %s", status)
        if status in ConductorStatusGroup.FAILED:
            eh.check_failed("Conductor ticket is broken")
        elif status in ConductorStatusGroup.DONE:
            return True
        return False

    @staticmethod
    def _get_ticket_id_from_response(response):
        m = TICKET_ID_REGEX.search(response)
        return m.group(1)

    def on_execute(self):
        conductor_api = Api(task=self, auth=OauthAuth(sdk2.Vault.data(VAULT_OWNER, VAULT_NAME)))
        retries_decorator = retries(max_tries=6, exceptions=(Timeout, HTTPError))
        ticket_add_wrapped = retries_decorator(conductor_api.ticket_add)
        ticket_status_wrapped = retries_decorator(conductor_api.ticket_status)

        with self.memoize_stage.ticket_creation:
            ticket_creation_params = dict(packages=self.Parameters.packages,
                                          branch=self.Parameters.branch,)

            if self.Parameters.filters:
                filter_len = len(self.Parameters.filters[0].split("/"))
                all_filters = []
                for filter in self.Parameters.filters:
                    parsed_filter = filter.split("/")
                    eh.verify(len(parsed_filter) == filter_len, "All filters must be the same length")
                    all_filters.append(parsed_filter)
                if filter_len == 2:
                    ticket_creation_params['workflows'] = all_filters
                if filter_len == 3:
                    ticket_creation_params['deploy_groups'] = all_filters

            ticket_id = ticket_add_wrapped(**ticket_creation_params)
            ticket_id = self._get_ticket_id_from_response(ticket_id)
            self.Parameters.ticked_id = ticket_id
            self.Parameters.ticket_url = "https://c.yandex-team.ru/tickets/" + ticket_id
            logging.info("Conductor ticket created with id %s", ticket_id)
            self.set_info(
                "Conductor URL: <a href=\"{0}\">{0}</a>".format(
                    self.Parameters.ticket_url,
                ),
                do_escape=False,
            )
            start_time = time.time()
            while (time.time() - start_time) < self.Parameters.active_waiting_duration:
                if self._check_status(ticket_id, ticket_status_wrapped):
                    return
                time.sleep(60.0)
            logging.info("Task went into passive waiting")
        with self.memoize_stage.waiting_ticket(self.Parameters.passive_waiting_tries) as st:
            ticket_id = self.Parameters.ticked_id
            if self._check_status(ticket_id, ticket_status_wrapped):
                return
            logging.info("%s try of passive waiting", st.runs)
            raise sdk2.WaitTime(self.Parameters.passive_waiting_duration)
        eh.fail("Ticket waiting timeout")
