# coding: utf-8

import logging
import json
import datetime
import requests
import time

from sandbox import sdk2

VAULT_OWNER = "BILLING-CI"
VAULT_NAME = "infra_token_robot_billing_ci"
INFRA_URL = 'https://infra-api.yandex-team.ru/v1/'


class EventCreationException(BaseException):
    pass


class CreateInfraEvent(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        service_id = sdk2.parameters.Integer(
            "Service ID",
            required=True,
        )
        environment_id = sdk2.parameters.Integer(
            "Environment ID",
            required=True,
        )
        datacenters = sdk2.parameters.Dict(
            "Affected datacenters {iva, man, myt, sas, vla : bool}",
        )
        event_type = sdk2.parameters.String(
            "Event type",
            required=True,
        )
        event_severity = sdk2.parameters.String(
            "Event severity",
            required=True,
        )
        title = sdk2.parameters.String(
            "Title",
            required=True,
        )
        description = sdk2.parameters.String(
            "Description",
        )
        tickets = sdk2.parameters.String(
            "Tickets",
        )
        start_time = sdk2.parameters.String(
            "Start time (YYYY-MM-DD HH24:MI:SS)",
        )
        finish_time = sdk2.parameters.String(
            "Finish time (YYYY-MM-DD HH24:MI:SS)",
        )
        meta = sdk2.parameters.String(
            "Meta info",
        )
        send_email_notification = sdk2.parameters.Bool(
            "Send email notification",
            default=True,
        )
        set_all_available_dc = sdk2.parameters.Bool(
            "Set all available dc",
            default=False
        )

        with sdk2.parameters.Output:
            event_id = sdk2.parameters.String(
                "Id of created event", required=True
            )

    def on_execute(self):

        def cast_dt(input):
            DT_FORMAT = '%Y-%m-%d %H:%M:%S'
            if input:
                return time.mktime((datetime.datetime.strptime(input, DT_FORMAT)).timetuple())
            return None

        infra_token = sdk2.Vault.data(VAULT_OWNER, VAULT_NAME)

        headers = {
            'Authorization': 'OAuth {}'.format(infra_token),
            'Content-Type': 'application/json',
        }

        data = {
            "title": self.Parameters.title,
            "description": self.Parameters.description,
            "environmentId": self.Parameters.environment_id,
            "serviceId": self.Parameters.service_id,
            "startTime": cast_dt(self.Parameters.start_time),
            "finishTime": cast_dt(self.Parameters.finish_time),
            "type": self.Parameters.event_type,
            "severity": self.Parameters.event_severity,
            "tickets": self.Parameters.tickets,
            "sendEmailNotifications": self.Parameters.send_email_notification,
            "setAllAvailableDc": self.Parameters.set_all_available_dc,
        }

        if self.Parameters.datacenters:
            data.update(self.Parameters.datacenters)

        if not self.Parameters.start_time:
            data['startTime'] = datetime.datetime.now()

        if self.Parameters.meta:
            data['meta'] = json.loads(self.Parameters.meta)

        logging.info("request data: {}".format(json.dumps(data)))

        response = requests.post(INFRA_URL + 'events',
                                 data=json.dumps(data),
                                 headers=headers,
                                 verify=False)

        if response.status_code == 200:
            result = response.json()
            event_id = result['id']
            self.Parameters.event_id = event_id
            return
        else:
            raise EventCreationException('Response status code: {}'.format(response.status_code))
