# -*- coding: utf-8 -*-

import jinja2
import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.utils import get_task_link
from sandbox.projects.balance.CreateConductorTicket import CreateConductorTicket
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

VAULT_OWNER = "BILLING-CI"
VAULT_NAME = "startrek_token"


class DoConductorRelease(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(rm_params.ComponentName2):
        packages = sdk2.parameters.List(
            "Ticket packages (comma separated)",
            required=True,
        )
        packages_version = sdk2.parameters.String(
            "Version of released packages",
            required=True,
        )
        ticket_filters = sdk2.parameters.List(
            "Ticket filters (semicolon separated)",
            required=True,
            description="same order as packages",
        )
        ticket_branches = sdk2.parameters.List(
            "Ticket branches",
            required=True,
            description="same order as packages",
        )
        with sdk2.parameters.Group("Order parameters") as order_group:
            ordered = sdk2.parameters.Bool(
                "Create deploy tickets in order",
                default=False,
                description="Do deploy in specific order",
            )
            order_list = sdk2.parameters.List(
                "Order for tickets",
                description="indexes of tickets (comma seperated if multiple tickets must be created at the same time)",
            )
        with sdk2.parameters.Group("Deploy") as deploy_group:
            with sdk2.parameters.String(
                "Deploy environment",
                required=False,
                description="timeline integration",
            ) as environment:
                environment.values.unstable = None
                environment.values.testing = environment.Value(default=True)
                environment.values.prestable = None
                environment.values.stable = None
                environment.values.hotfix = None
            ticket_templates = sdk2.parameters.List(
                "Ticket template names",
                description="This is used to render jinja template",
            )
            dont_create_comment = sdk2.parameters.Bool(
                "Don't create comment or ticket in Startrek",
                default=False,
            )
        release_number = sdk2.parameters.Integer("Release number", required=False)
        major_release_num = sdk2.parameters.Integer("Major release number", required=False)
        minor_release_num = sdk2.parameters.Integer("Minor release number", required=False)

    class Context(sdk2.Task.Context):
        current_order = 0
        subtask_ids = []

    def _create_subtask(self, packages, branch, filters):
        return CreateConductorTicket(
            self,
            description="Create ticket for DoConductorRelease task #{}: \"{}\"".format(self.id,
                                                                                       self.Parameters.description),
            owner=self.owner,
            priority=self.Parameters.priority,
            packages=packages,
            branch=branch,
            filters=filters,
            active_waiting_duration=60 * 40,
            passive_waiting_tries=20,
            passive_waiting_duration=3600,
        )

    def _process_task_in_async_mode(self):
        with self.memoize_stage.tickets_creation:
            subtasks = []
            for packages, filter, branch in zip(self.Parameters.packages, self.Parameters.ticket_filters,
                                                self.Parameters.ticket_branches):
                subtask = self._create_subtask(
                    packages={package: self.Parameters.packages_version for package in packages.split(",")},
                    branch=branch,
                    filters=filter.split(";"),
                )
                subtask.enqueue()
                logging.info("Subtask is created and enqueued with id %s", subtask.id)
                subtasks.append(subtask)
                self.Context.subtask_ids.append(subtask.id)
            targets = {subtask.id: 'ticket_url' for subtask in subtasks}
            raise sdk2.WaitOutput(targets, wait_all=True, timeout=60 * 10)
        with self.memoize_stage.ticket_processing:
            subtasks = self.find()
            if self.Parameters.environment:
                self._create_timeline_comment(self.Parameters.environment)
            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

    def _parse_ticket_order(self, indexes):
        return (
            (self.Parameters.packages[idx], self.Parameters.ticket_filters[idx], self.Parameters.ticket_branches[idx])
            for idx in map(int, indexes.split(","))
        )

    def _process_task_in_ordered_mode(self):
        if self.Context.current_order < len(self.Parameters.order_list):
            utils.check_subtasks_fails()
            indexes = self.Parameters.order_list[self.Context.current_order]
            logging.info("Start of order %s at index %s", indexes, self.Context.current_order)
            subtasks = []
            for packages, filter, branch in self._parse_ticket_order(indexes):
                subtask = self._create_subtask(
                    packages={package: self.Parameters.packages_version for package in packages.split(",")},
                    branch=branch,
                    filters=filter.split(";"),
                )
                subtask.enqueue()
                logging.info("Subtask is created and enqueued with id %s", subtask.id)
                subtasks.append(subtask)
            self.Context.current_order += 1
            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.ticket_processing:
            if self.Parameters.environment:
                self._create_timeline_comment(self.Parameters.environment)

    def on_execute(self):
        eh.ensure(
            (
                len(self.Parameters.packages) ==
                len(self.Parameters.ticket_filters) ==
                len(self.Parameters.ticket_branches)
            ),
            "Packages, ticket filters and branches must have same length"
        )
        if self.Parameters.ordered:
            self._process_task_in_ordered_mode()
        else:
            self._process_task_in_async_mode()
        subtasks = self.find()
        subtasks_info = []
        for subtask in subtasks:
            subtasks_info.append(
                "<a href=\"{0}\">{0}</a>".format(subtask.Parameters.ticket_url) + utils.colored_status(subtask.status)
            )
        self.set_info(
            "Subtasks info:\n" + "\n".join(subtasks_info),
            do_escape=False,
        )
        utils.check_subtasks_fails()

    @staticmethod
    def _read_template_file(filename):
        template_path = os.path.join(os.path.dirname(__file__), filename)
        return jinja2.Template(common.fs.read_file(template_path))

    @staticmethod
    def _create_hotfix_issue(st_helper, c_info, description, version, parent):  # TODO: move this to st_helper
        kwargs = {
            "queue": c_info.notify_cfg__st__queue,
            "summary": u"Выкладка хотфикса {}".format(version),
            "description": description,
            "tags": "hotfix",
            "type": {"name": u"Хотфикс"},
            "parent": parent,
        }
        return st_helper.st_client.issues.create(**kwargs)

    def _create_timeline_comment(self, env):
        if self.Parameters.dont_create_comment:
            logging.info("Skip creating startrek comment")
            return
        st_helper = STHelper(sdk2.Vault.data(VAULT_OWNER, VAULT_NAME))
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        issue = st_helper.find_ticket_by_release_number(self.Parameters.release_number, c_info)

        template_data = {
            "major_release_num": self.Parameters.major_release_num,
            "minor_release_num": self.Parameters.minor_release_num,
            "packages_version": self.Parameters.packages_version,
            "packages_links": {
                ",".join(subtask.Parameters.packages.iterkeys()): subtask.Parameters.ticket_url
                for subtask in self.find()
            },
            "env": env,
            "sandbox_link": get_task_link(self.id),
        }
        if self.Context.subtask_ids:
            for idx, task_id in enumerate(self.Context.subtask_ids):
                ticket_template = self.Parameters.ticket_templates[idx]
                ticket_link = sdk2.Task[task_id].Parameters.ticket_url
                if ticket_template not in template_data:
                    template_data[ticket_template] = ticket_link or "!!NOT CREATED!!"
                else:
                    template_data[ticket_template] += " " + ticket_link

        logging.info("Creating ST comment for %s", env)
        if env == "stable":
            template = self._read_template_file("prod_ticket_template.txt")
            st_helper.create_comment(issue, template.render(template_data))
        elif env == "prestable":
            template = self._read_template_file("preprod_ticket_template.txt")
            st_helper.create_comment(issue, template.render(template_data))
        elif env == "hotfix":
            template = self._read_template_file("hotfix_ticket_template.txt")
            hotfix_issue = self._create_hotfix_issue(
                st_helper=st_helper,
                c_info=c_info,
                description=template.render(template_data),
                version=self.Parameters.packages_version,
                parent=issue,
            )
            hotfix_issue.transitions['testing'].execute()
        elif env == "testing":
            st_helper.create_comment(issue, "Conductor tickets with version {} deployed to {} by (({} {}))".format(
                self.Parameters.packages_version,
                env,
                get_task_link(self.id),
                "sandbox task",
            ))
