import logging
import re
import requests
import time
import xml.etree.ElementTree as et

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import error_handlers as eh

TEAMCITY_API_URL = "https://teamcity.yandex-team.ru"
VAULT_OWNER = "BILLING-CI"
VAULT_NAME = "teamcity_token"

ARCADIA_REGEX = re.compile(r"arcadia:\/arc\/(.*?)\/arcadia@([\d]*)")

DEFAULT_BUILD_TIMEOUT = 60 * 10  # 10 minutes


class RunTeamcityBuild(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        build_task_id = sdk2.parameters.String("Id of build task in Teamcity", default="Billing_Tools_Temp_BuildToolsFromArcadia", required=True)
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        arcadia_patch = sdk2.parameters.String("Arcadia patch")
        arcanum_review_id = sdk2.parameters.Integer("Arcanum review id")
        package_version = sdk2.parameters.String("Package version")
        build_timeout = sdk2.parameters.Integer("Build timeout in seconds", default=DEFAULT_BUILD_TIMEOUT)
        teamcity_parameters = sdk2.parameters.Dict(
            "Job parameters",
            default={},
            description=(
                "You can use format templates in values\n"
                " - {pr_id} pull request id\n"
                " - {svn_branch} branch path in terms of teamcity logic\n"
                " - {revision} revision\n"
            )
        )
        build_uid = sdk2.parameters.String(
            "Unique id of this build (leave empty if uniqueness is not required)",
            default="",
            description="This uid is used for reusing build results in case of multiple running tasks"
        )
        with sdk2.parameters.Group("Unit Tests") as unit_tests_parameters:
            skip_if_patch_empty = sdk2.parameters.Bool(
                "Skip if arcadia patch is empty",
                default=False,
                description="This parameter is used to skip tests on unpatched trunk",
            )

        with sdk2.parameters.Output:
            teamcity_url = sdk2.parameters.String(
                "Url of created task", required=True
            )

    class Context(sdk2.Task.Context):
        arcanum_review_id = 0

    def _get_teamcity_build_xml(self, build_type, properties):
        build = et.Element("build")
        et.SubElement(build, "buildType", id=build_type)
        properties_xml = et.SubElement(build, "properties")
        for name, value in properties.iteritems():
            et.SubElement(properties_xml, "property", name=name, value=value)
        return et.tostring(build, encoding="utf-8")

    def on_enqueue(self):
        if self.Parameters.build_uid:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(
                        name="{}_{}".format(self.type, self.Parameters.build_uid),
                        weight=1,
                        capacity=1,
                    )
                ],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
            )

    def _check_existing_tasks(self):
        task = RunTeamcityBuild.find(
            status=(ctt.Status.SUCCESS,),
            input_parameters={'build_uid': self.Parameters.build_uid}
        ).first()
        if task:
            logging.info("Found task with id %s", task.id)
        return task is not None

    def on_execute(self):
        if self.Parameters.build_uid:
            if self._check_existing_tasks():
                self.set_info("Skipping execution, because success task with same build_uid is found")
                logging.info("Skipping execution, because success task is found")
                return
        if self.Parameters.skip_if_patch_empty and not self.Parameters.arcadia_patch:
            logging.info("Skipping task on unpatched trunk")
            self.set_info("Skipping task on unpatched trunk")
            return
        headers = {
            "Authorization": "OAuth {}".format(sdk2.Vault.data(VAULT_OWNER, VAULT_NAME)),
            "Content-Type": "application/xml",
        }
        match = re.search(ARCADIA_REGEX, self.Parameters.checkout_arcadia_from_url)
        eh.ensure(match, "Can't parse arcadia url. It must be arcadia:/arc/.../arcadia@rev")
        arcadia_branch_for_tc = match.group(1)
        arcadia_revision = match.group(2)
        props = {
            key: value.format(
                pr_id=self.Context.arcanum_review_id,
                svn_branch=arcadia_branch_for_tc,
                revision=arcadia_revision,
            )
            for key, value in self.Parameters.teamcity_parameters.iteritems()
        }

        request_data = self._get_teamcity_build_xml(self.Parameters.build_task_id, props)
        logging.info("Request data: %s", request_data)
        response = requests.post(TEAMCITY_API_URL + "/app/rest/buildQueue", headers=headers, data=request_data).text
        logging.info("Server response: %s", response)
        root = et.fromstring(response)
        task_url = root.attrib.get("webUrl")
        task_href = root.attrib["href"]
        if task_url:
            self.Parameters.teamcity_url = task_url
            self.set_info("Teamcity URL: <a href=\"{0}\">{0}</a>".format(task_url), do_escape=False)
        else:
            self.set_info("Can't get Teamcity URL")

        build_start = time.time()
        while (time.time() - build_start) < self.Parameters.build_timeout:
            response = requests.get(TEAMCITY_API_URL + task_href, headers=headers).text
            root = et.fromstring(response)
            task_state = root.attrib["state"]
            if task_state == "finished":
                task_status = root.attrib["status"]
                eh.ensure(task_status == "SUCCESS", "Teamcity task status is {}".format(task_status))
                artifacts_href = root.find("artifacts").attrib["href"]

                response = requests.get(TEAMCITY_API_URL + artifacts_href, headers=headers).text
                root = et.fromstring(response)
                all_tickets = []
                for fl in root.findall("file"):
                    tickets = requests.get(TEAMCITY_API_URL + fl.attrib["href"], headers=headers).text.strip().split("\n")
                    all_tickets.extend(tickets)
                self.set_info("Ticket list:\n" + "\n".join(all_tickets))
                return
            time.sleep(60.0)
        eh.check_failed("Teamcity build timeout")
