# --*-- encoding: utf-8 --*--
import logging
import re
from collections import namedtuple
from decimal import Decimal
from os import path, environ
from textwrap import dedent
from uuid import uuid4

from jinja2 import Environment, BaseLoader

try:
    from sandbox import common
    from sandbox import sdk2
    from sandbox.projects.common.arcadia import sdk as arcadia_sdk
    from sandbox.sdk2 import yav
    from sandbox.sdk2.helpers import subprocess, ProcessLog

except ImportError:
    # Presumably this is a local run (using __main__) for debugging purposes.
    from unittest.mock import MagicMock
    sdk2 = MagicMock()


MARKER_PY3 = '<py3test>'
MARKER_PY2 = '<pytest>'


class RunTestsManual(sdk2.Task):
    """Simple generic task which allows manual test runs, exposing results via task context and Sandbox UI.

    Task is aimed to run and print out results for regression (large) tests
    using Python's pytest.

    Arcadia sources are fetched using Arc.
    Requires ARC_TOKEN to be present in environment or Sandbox Vault.

    """
    class Parameters(sdk2.Task.Parameters):

        description = 'Run tests and show results.'
        fail_on_any_error = True

        with sdk2.parameters.Group('Testing') as group_tests:

            source = sdk2.parameters.String(
                'Source path',
                description='Source path to run tests against. E.g.: billing/bcl/src/tests/regression',
                required=True
            )

            branch = sdk2.parameters.String(
                'Source branch',
                description='Arcadia branch to build and test. E.g.: trunk, users/idlesign/somebranch',
                default='trunk',
                required=True
            )

            env_secrets = sdk2.parameters.Dict(
                'Secrets for environment',
                description=(
                    'Maps YAV secret to environment variable name. '
                    '`<sec-id>@<key> = <env-var>`: sec-xxxx@myvalue = MY_TOKEN '
                    '!!! Secret MUST be delegated to Sandbox. '
                    'Use `delegate_secret` field below or see https://nda.ya.ru/t/RUuvxrYx4YL3Lq'
                )
            )

            secret = sdk2.parameters.YavSecret(
                'Secret to delegate',
                description='A YAV secret to delegate to Sandbox. Used solely to simplify delegation process.'
            )

            sandbox_resources_ids = sdk2.parameters.List(
                'Sandbox resources',
                description='Download and extract sandbox resources with specified ids'
            )

            env = sdk2.parameters.Dict(
                'Environment variables',
                description='Map specified value to environment variable name. '
                            'To set a path of extracted resource as a value for a variable, '
                            'use `res:<res-id>`. E.g.: `res:12345678`'
            )

    class Context(sdk2.Context):

        report_tests = {}

    @staticmethod
    def initialize_secrets(mapping):
        """Get secret data from YAV by mapping and put it into environment vars.

        :param mapping: sec-id@key = VARNAME

        """
        by_secret = {}
        SecretMeta = namedtuple('SecretMeta', ['alias', 'entries'])
        SecretEntry = namedtuple('SecretEntry', ['key', 'alias'])

        for name_secret, name_env in mapping.items():
            secret_id, _, secret_key = name_secret.partition('@')
            name_env = (name_env or secret_key).upper()

            logging.debug(
                'Secret mapping: id %s, key %s -> %s',
                secret_id, secret_key, name_env)

            secret_tuple = SecretEntry(secret_key, name_env)

            if secret_id in by_secret:
                by_secret[secret_id].entries.append(secret_tuple)

            else:
                by_secret[secret_id] = SecretMeta(
                    alias='sec%s' % (len(by_secret) + 1),
                    entries=[secret_tuple]
                )

        call_kwargs = {
            secret_meta.alias: yav.Secret(secret_id)
            for secret_id, secret_meta in by_secret.items()}

        if not call_kwargs:
            return

        secrets = yav.Yav(**call_kwargs)

        for secret_id, secret_meta in by_secret.items():
            secret_data = getattr(secrets, secret_meta.alias, None)

            if secret_data is None:
                logging.warning('No secret data from YAV for %s', secret_id)
                continue

            for (name_key, name_env) in secret_meta.entries:
                value = secret_data.get(name_key, None)

                if value is None:
                    logging.warning(
                        'No secret data from YAV for %s@%s', secret_id, name_key)
                    continue

                environ[name_env] = value

    @staticmethod
    def download_and_extract_resources(resource_ids, env_maping):
        paths = dict()
        for resource_id in resource_ids:
            resource_id = int(resource_id)
            resource = sdk2.Resource[resource_id]
            rdata = sdk2.ResourceData(resource)
            rpath = rdata.path  # See https://docs.python.org/dev/library/pathlib.html
            paths[resource_id] = rpath
        for var_name, value in env_maping.items():
            if value.startswith('res:'):
                resource_id = int(value[4:])
                environ[var_name] = str(paths[resource_id])
            else:
                environ[var_name] = value

    def on_execute(self):

        params = self.Parameters

        branch = 'arcadia-arc:/#%s' % params.branch
        source = str(params.source)

        logging.info('Manual test for: %s %s', branch, source)

        with arcadia_sdk.mount_arc_path(branch) as path_arc:

            path_ya = path.join(path_arc, 'ya')
            path_source = path.join(path_arc, source)

            self.initialize_secrets(params.env_secrets)
            self.download_and_extract_resources(params.sandbox_resources_ids, params.env)

            with ProcessLog(self, logger='tests') as log:

                process = subprocess.Popen(
                    '%s make -ttt --show-passed-tests --no-src-links' % path_ya,
                    shell=True,
                    stdout=log.stdout,
                    stderr=subprocess.STDOUT,
                    cwd=path_source,
                )
                process.wait()

                with open(str(log.stdout.path), 'r') as logfile:
                    out = logfile.read()

                if process.returncode and not (MARKER_PY2 in out or MARKER_PY3 in out):
                    raise common.errors.TaskFailure('Running test command failed, see logs (e.g. tests.out.log).')

        parsed = self.report_parse(out)

        logging.debug('Saving context ...')
        context = self.Context
        context.report_tests = parsed
        context.save()

        if process.returncode:
            raise common.errors.TaskFailure('Some tests failed. See results on "Tests Result" tab.')

    @sdk2.report(title='Tests Result', label='tests')
    def report(self):
        data = self.Context.report_tests

        if not data:
            return '<b>No test data available in task context</b>'

        return self.report_build(data)

    @staticmethod
    def report_parse(data):
        logging.debug('Parsing report output ...')
        return PytestLogParser.parse(data)

    @classmethod
    def report_build(cls, data):
        logging.debug('Building report HTML ...')
        return ReportRenderer.render(data)


class PytestLogParser(object):

    @staticmethod
    def parse(data):
        _, _, rest = data.partition('%s [' % (MARKER_PY3 if MARKER_PY3 in data else MARKER_PY2))
        rest, _, _ = rest.partition('Info: Node count')

        def empty_current_item():
            current_item.clear()
            current_item.update({
                'status': '',
                'file': '',
                'name': '',
                'params': '',
                'tags': '',
                'env': '',
                'time': 0,
                'trace': [],
                'log_url': '',
            })

        def finalize_current_item():
            tests.append(current_item.copy())
            empty_current_item()

        current_item = {}
        empty_current_item()

        tests = []
        footer = []
        in_footer = False

        re_test_line = re.compile(
            r'\[(fail|xfail|good|xpass)\]\s*'  # status
            r'([^:]+)::'  # file
            r'([^[]+)'  # name
            r'(\[([^]]+)\])?\s*'  # params
            r'(\[([^]]+)\])?\s*'  # tags
            r'\[([^]]+)\]\s'  # env
            r'\(([^\s]+)s\)'  # time
        )
        """
        Samples:
        [good] payment_systems.test_regr_payoneer.py::test_send_and_check_payment[123321123-3001.00-RUB-10303] [tags: regression, parametrize] [default-linux-x86_64-debug] (6.24s)
        [good] test_st.py::TestSt::test_exists [default-linux-x86_64-debug] (0.01s)
        """

        for line in rest.splitlines()[1:]:

            if in_footer:

                if not line.startswith('\t'):
                    # end of summary block
                    break

                footer.append(line.strip())

                continue

            if line.startswith('Logsdir:') or line.startswith('------'):
                continue

            if line.startswith('Total ') and line.endswith(' tests:'):
                in_footer = True
                continue

            if line.startswith('['):
                match = re_test_line.match(line)

                if match:
                    try:
                        status = match.group(1).lower()

                        current_item.update({
                            'status': status,
                            'file': match.group(2),
                            'name': match.group(3),
                            'params': match.group(5) or '',
                            'tags': match.group(7) or '',
                            'env': match.group(8),
                            'time': match.group(9),
                        })

                        if status == 'xfail':
                            finalize_current_item()

                    except IndexError:
                        raise IndexError('Line group mismatch: %s' % line)

                continue

            if line.startswith('Log:'):
                # Завершающая строка блока по тесту.
                current_item['log_url'] = line.replace('Log: ', '', 1)
                finalize_current_item()
                continue

            current_item['trace'].append(line)

        summary = {}
        if tests:
            for summary_item in footer:
                count, _, status = summary_item.partition(' - ')
                summary[status.strip().lower()] = count.strip()

        def sort(items):
            return sorted(items, key=lambda item: (item['name'], item['file']))

        tests_good = [test for test in tests if test['status'] == 'good']
        tests_bad = [test for test in tests if test['status'] != 'good']
        tests_slow = sorted(tests, key=lambda item: Decimal(item['time']), reverse=True)[:10]

        parsed = {
            'summary': summary,
            'tests_good': sort(tests_good),
            'tests_bad': sort(tests_bad),
            'tests_slow': tests_slow,
        }
        return parsed


class ReportRenderer(object):

    duration_units = {
        'нед.': 60 * 60 * 24 * 7,
        'дн.': 60 * 60 * 24,
        'ч.': 60 * 60,
        'мин.': 60,
        'сек.': 1,
    }

    @classmethod
    def humanize_duration(cls, val):

        if not val:
            return ''

        chunks = []

        for unit, div in cls.duration_units.items():
            count, val = divmod(int(Decimal(val)), div)

            if count > 0:
                chunks.append('%s %s' % (count, unit))

        return ' '.join(chunks)

    @classmethod
    def _render(cls, template, context):
        env = Environment(loader=BaseLoader())
        env.filters['human_duration'] = cls.humanize_duration
        template = env.from_string(template)
        context.update({
            'uuid4': uuid4,
        })
        rendered = template.render(context)
        return rendered

    @classmethod
    def render(cls, data):
        rendered = cls._render(template=cls.PAGE_TPL, context=data)
        return rendered

    PAGE_TPL = dedent('''
    <link rel="stylesheet" href="https://yastatic.net/bootstrap/4.4.1/css/bootstrap.min.css">

    {% macro status_badge(status, count=0) -%}
        <span class="badge
         badge-{% if status == 'fail' %}danger{% elif status == 'good' %}success{% else %}warning{% endif %}">
            {{ status }}{% if count %} {{ count }}{% endif %}
        </span>
    {%- endmacro %}
    <div class="mb-5">
        <h3 class="text-uppercase">Summary</h3>
        <div>
        {% for in_status, in_count in summary.items() %}
            {{ status_badge(in_status, in_count) }}
        {% endfor %}
        </div>
    </div>

    <div class="mb-5">
        <h3 class="text-uppercase">Failed</h3>
        <table class="table table-hover">
        {% for test in tests_bad %}
        <tr>
            <td><span class="mr-2">{{ status_badge(test['status']) }}</span></td>
            <td>
                <span class="mr-3">{{ test['name'] }}</span><br>
                <span class="text-muted small">{{ test['file'] }}</span>
            </td>
            <td>
                <div class="mb-3"><span class="small">{{ test['params'] }}</span></div>
                <div><pre class="small text-muted"><code>{{ '\n'.join(test['trace']) }}</code></pre></div>
            </td>
            <td width="1"><sub>{{ test['time'] }}</sub></td>
        </tr>
        {% endfor %}
        </table>
    </div>

    <div class="mb-5">
        <h3 class="text-uppercase">Slowest</h3>
        <table class="table table-hover">
        {% for test in tests_slow %}
        <tr>
            <td>{{ test['name'] }}</td>
            <td>
                <span class="small">{{ test['file'] }}</span><br>
                <span class="text-muted small">{{ test['params'] }}</span>
            </td>
            <td><span class="small">{{ test['time'] }}</span></td>
        </tr>
        {% endfor %}
        </table>
    </div>
    ''')


if __name__ == '__main__':

    filepath = 'tests.out.log'

    with open(filepath, 'r') as f:
        data = f.read()

    parsed = PytestLogParser.parse(data)
    rendered = ReportRenderer.render(parsed)

    with open('tests.html', 'w') as f:
        f.write(rendered)
