# -*- coding: utf-8 -*-
import os

from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.balancer import resources as br
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.balancer.task import BaseBalancerTask


class BuildCorelimitTool(sp.SandboxBoolParameter):
    # MINOTAUR-1295
    name = 'build_corelimit_tool'
    description = 'Build corelimit tool'
    default_value = False


class BuildTicketsDeployTool(sp.SandboxBoolParameter):
    name = 'build_tickets_deploy_tool'
    description = 'Build tickets_deploy tool'
    default_value = False


class BuildBalancerBundle(nanny.ReleaseToNannyTask, BaseBalancerTask):
    """
        Build balancer binary
    """

    type = 'BUILD_BALANCER_BUNDLE'

    execution_space = 45000

    input_parameters = BaseBalancerTask.input_parameters + [
        build_params.ArcadiaPatch,
        build_params.LTO,
        build_params.ThinLTO,
        BuildCorelimitTool,
        BuildTicketsDeployTool,
    ]

    def initCtx(self):
        BaseBalancerTask.initCtx(self)
        self.ctx['kill_timeout'] = 60 * 80

    def get_targets(self):
        targets = ['balancer/daemons/balancer']

        if utils.get_or_default(self.ctx, BuildCorelimitTool):
            targets.append('balancer/production/x/corelimit')

        if utils.get_or_default(self.ctx, BuildTicketsDeployTool):
            targets.append('balancer/production/tickets_deploy')

        return targets

    def _add_resource_spec(self, resources, resource_type):
        resources[resource_type.name] = {
            'description': self.descr,
            'resource_type': resource_type,
            'resource_path': os.path.basename(resource_type.arcadia_build_path),
        }

    def _art_spec(self, resource_type):
        return {
            'path': resource_type.arcadia_build_path,
        }

    def get_resources(self):
        if self.ctx['sanitize'] == "address":
            balancer = br.BALANCER_EXECUTABLE_ASAN
        elif self.ctx['sanitize'] == "leak":
            balancer = br.BALANCER_EXECUTABLE_LSAN
        elif self.ctx['sanitize'] == "memory":
            balancer = br.BALANCER_EXECUTABLE_MSAN
        elif self.ctx['build_type'] == "debug":
            balancer = br.BALANCER_EXECUTABLE_DEBUG
        else:
            balancer = br.BALANCER_EXECUTABLE

        resources = {}
        self._add_resource_spec(resources, balancer)

        if utils.get_or_default(self.ctx, BuildCorelimitTool):
            self._add_resource_spec(resources, br.CORELIMIT_EXECUTABLE)

        if utils.get_or_default(self.ctx, BuildTicketsDeployTool):
            self._add_resource_spec(resources, br.TICKETS_DEPLOY_EXECUTABLE)

        return resources

    def get_arts(self):
        arts = [self._art_spec(br.BALANCER_EXECUTABLE)]

        if utils.get_or_default(self.ctx, BuildCorelimitTool):
            arts.append(self._art_spec(br.CORELIMIT_EXECUTABLE))

        if utils.get_or_default(self.ctx, BuildTicketsDeployTool):
            arts.append(self._art_spec(br.TICKETS_DEPLOY_EXECUTABLE))

        return arts


__Task__ = BuildBalancerBundle
