# coding: utf-8

import os
import time
import copy
import sandbox.sandboxsdk.process as sdk_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.common.balancer import base_balancer_configs
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.configs.balancer_config import BalancerConfigCfg
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper


class EnableGenCfgCache(sp.SandboxBoolParameter):
    name = 'enable_gencfg_cache'
    description = 'Enable GenCFG instances cache'
    default_value = True
    required = False


class NannyTokenVaultName(sp.SandboxStringParameter):
    name = 'nanny_token_vault_name'
    description = "Nanny OAuth token vault name"
    default_value = "nanny_oauth_token"
    required = False


class NannyTokenVaultOwner(sp.SandboxStringParameter):
    name = 'nanny_token_vault_owner'
    description = "Nanny OAuth token vault owner"
    default_value = "robot-balancer-tests"
    required = False


@rm_notify.notify2()
class Task(base_balancer_configs.BaseTask):
    """Generate, test and release tarball with L7 balancer configs"""

    type = 'BUILD_BALANCER_CONFIGS_L7'

    base_description_text = "Build and test L7 balancer's configs in task"

    input_parameters = copy.copy(base_balancer_configs.BaseTask.input_parameters)
    input_parameters.extend([
        EnableGenCfgCache,
        NannyTokenVaultName,
        NannyTokenVaultOwner,
    ])

    def initCtx(self):
        self.ctx['kill_timeout'] = 60 * 40

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, BalancerConfigCfg.name)
        base_balancer_configs.BaseTask.on_enqueue(self)

    def _run(self, args, log_prefix):
        sdk_process.run_process(
            args,
            work_dir=self.generator_root,
            log_prefix=log_prefix,
            environment=self.generator_env,
        )

    def _calc_generation_time(self, generation_start_time):
        generation_time = int(time.time() - generation_start_time)
        self.ctx["generation_time"] = generation_time
        self.set_info("Configs generation time: {}s".format(generation_time))

    def run_build(self):
        assert self.ctx.get(base_balancer_configs.YAMAKE_TASK_ID)

        my_env = os.environ.copy()

        nanny_token_vault_name = utils.get_or_default(self.ctx, NannyTokenVaultName)
        nanny_token_vault_owner = utils.get_or_default(self.ctx, NannyTokenVaultOwner)
        nanny_token = self.get_vault_data(nanny_token_vault_owner, nanny_token_vault_name)
        assert isinstance(nanny_token, str) and len(nanny_token) > 0, 'Bad nanny token: {}'.format(nanny_token)
        my_env['NANNY_OAUTH_TOKEN'] = nanny_token

        yamake_task = channel.sandbox.get_task(self.ctx[base_balancer_configs.YAMAKE_TASK_ID])
        self.ctx['generator_binary_id'] = yamake_task.ctx['ap_packs']['project']
        generator_path = channel.task.sync_resource(self.ctx['generator_binary_id'])
        my_env['CUSTOM_GENPY'] = os.path.join(generator_path, 'utils')
        # Be verbose. Always, I said
        my_env['GENPY_VERBOSE'] = "--verbose --verbose"
        my_env["DISABLE_TRUNK_FALLBACK"] = "--disable-trunk-fallback"

        if utils.get_or_default(self.ctx, EnableGenCfgCache):
            my_env['GENPY_ENABLE_CACHE'] = "--enable-cache"

        self.generator_env = my_env

        generation_start_time = time.time()

        self._run(['pwd'], log_prefix='pwd')
        self._run(['ls', '-R'], log_prefix='ls')

        # production configs
        self._run(
            ['bash', './gen-configs-l7-balancer.sh', 'api'],
            log_prefix='generate_l7',
        )
        # load configs
        self._run(
            ['bash', './gen-balancer-test.sh', 'api'],
            log_prefix='generate_test',
        )
        self._calc_generation_time(generation_start_time)
        self._run(
            ['tar', '-czf', os.path.join(os.path.abspath(''), 'balancer_configs_l7.tgz'), 'generated/l7-balancer'],
            log_prefix='tar_configs_l7',
        )
        self._run(
            ['cp', '-r', 'generated/l7-balancer', os.path.join(os.path.abspath(''), 'configs_l7')],
            log_prefix='copy_dir_configs_l7',
        )
        self.create_resource(
            arch='any',
            attributes={'ttl': 30},
            description=self.descr,
            resource_path='configs_l7',
            resource_type='BALANCER_GENCFG_CONFIGS_L7_DIR',
        )
        balancer_configs_l7 = self.create_resource(
            arch='any',
            attributes={'ttl': 30},
            description=self.descr,
            resource_path='balancer_configs_l7.tgz',
            resource_type='BALANCER_GENCFG_CONFIGS_L7_TGZ',
        )
        self.test_ctx['balancer_gencfg_configs_l7_id'] = balancer_configs_l7.id
        self.mark_resource_ready(balancer_configs_l7)

        # production yav-deploy.conf
        self._run(
            ['cp', './configs/l7heavy/yav-deploy.conf', os.path.join(os.path.abspath(''), 'yav-deploy.conf')],
            log_prefix='cp_yav_deploy'
        )
        balancer_yav_deploy_conf = self.create_resource(
            arch='any',
            attributes={'ttl': 30},
            description=self.descr,
            resource_path='yav-deploy.conf',
            resource_type='BALANCER_L7_YAV_DEPLOY_CONF',
        )
        self.mark_resource_ready(balancer_yav_deploy_conf)

        # load configs
        self._run(
            [
                'cp',
                'generated/l7-balancer/l7heavy_production_tun_man.cfg',
                'generated/load_testing/l7heavy_production_tun_man.cfg',
            ],
            log_prefix='copy_prod_config_for_load',
        )
        self._run(
            [
                'cp',
                'generated/l7-balancer/l7heavy_service_search_man.cfg',
                'generated/load_testing/l7heavy_service_search_man.cfg',
            ],
            log_prefix='copy_service_prod_config_for_load',
        )
        self._run(
            [
                'tar',
                '-czf',
                os.path.join(os.path.abspath(''), 'balancer_configs_test.tgz'),
                'generated/load_testing',
            ],
            log_prefix='tar_configs_test',
        )
        self._run(
            ['cp', '-r', 'generated/load_testing', os.path.join(os.path.abspath(''), 'configs_test')],
            log_prefix='copy_dir_configs_test',
        )
        balancer_configs_test_dir = self.create_resource(
            arch='any',
            attributes={'ttl': 1},
            description=self.descr,
            resource_path='configs_test',
            resource_type='BALANCER_GENCFG_CONFIGS_TEST_DIR',
        )
        balancer_configs_test_tgz = self.create_resource(
            arch='any',
            attributes={'ttl': 1},
            description=self.descr,
            resource_path='balancer_configs_test.tgz',
            resource_type='BALANCER_GENCFG_CONFIGS_TEST_TGZ',
        )
        self.mark_resource_ready(balancer_configs_test_dir)
        self.mark_resource_ready(balancer_configs_test_tgz)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = Task
