# -*- coding: utf-8 -*-
import os
import shutil
import logging
import multiprocessing
import difflib


import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2

from sandbox.common import fs

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2 as u2
from sandbox.projects.common.search import bugbanner2
from sandbox.projects.common.constants.constants import ARCADIA_URL_KEY
from sandbox.projects.balancer import resources as br
from sandbox.projects.websearch.begemot.tasks.BegemotCreateResponsesDiff import jsondiff
from sandbox.projects.balancer.CompareBalancerConfigs import config_grouping
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.common.sdk_compat import task_helper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.balancer_config import BalancerConfigCfg
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.sdk2.helpers import subprocess
import sandbox.projects.common.link_builder as lb


class BalancerConfigsDiff(sdk2.Resource):
    pass


class BalancerConfigsJson(sdk2.Resource):
    pass


def _lua_to_json_thread(args):
    """
    Convert lua config to json using lua_to_json binary
    (runs in separate process)
    """
    logging.info("lua_to_json for %s", args)
    try:
        subprocess.check_output(args)
    except Exception as exc:
        eh.log_exception("Exception in _lua_to_json_thread", exc)


def _jsondiff_thread(args):
    """
    Compare old json config with new json config using pyos@ jsondiff
    (runs in separate process)
    """
    file, old, new = args
    logging.info("Comparing %s", file)
    try:
        return file, jsondiff.diff(old, new)
    except Exception as exc:
        eh.log_exception("Exception in _jsondiff_thread", exc)


def _kdiff_thread(simplified_content):
    """
    Compare simplified representation of configs
    using kulikov@-mvel@ generalization kdiff tool
    @param simplified_content: dict with params
    """
    try:
        file = simplified_content["file"]
        old_input = simplified_content["old_input"]
        old_output = simplified_content["old_output"]
        new_input = simplified_content["new_input"]
        new_output = simplified_content["new_output"]
        kdiff_uses_argparse = simplified_content["kdiff_uses_argparse"]
        hash_markers = simplified_content["hash_markers"]

        main_args = ["python", "kdiff/simplify.py"]
        if hash_markers:
            main_args.append("--hash-markers")

        logging.info("Applying simplify for %s: %s -> %s", file, old_input, old_output)
        old_args = ["--input", old_input, "--output", old_output] if kdiff_uses_argparse else [old_input, old_output]
        subprocess.check_output(main_args + old_args)

        logging.info("Applying simplify for %s: %s -> %s", file, new_input, new_output)
        new_args = ["--input", new_input, "--output", new_output] if kdiff_uses_argparse else [new_input, new_output]
        subprocess.check_output(main_args + new_args)

        logging.info("Applying diff2html for %s: %s vs %s", file, old_output, new_output)

        diff_basename = os.path.join("diffs", file + ".simplified")
        diff_output = diff_basename + ".diff"
        # intermediate results before links post-processing
        diff_html_int = diff_basename + ".int.html"
        diff_html = diff_basename + ".html"
        subprocess.call(
            [
                "diff",
                "-ub",
                old_output,
                new_output,
            ],
            stdout=open(diff_output, 'w'),
        )

        has_diff = False
        if os.path.getsize(diff_output) > 0:
            has_diff = True
            # we process only non-empty diffs
            subprocess.check_output(
                [
                    "python",
                    "kdiff/diff2html.py",
                    "-i", diff_output,
                    "-o", diff_html_int,
                    "-a", "1",  # algo: word diff
                ],
            )
            subprocess.check_output(
                [
                    "python",
                    "kdiff/diff_postprocess.py",
                    "-i", diff_html_int,
                    "-o", diff_html,
                ],
            )
            # remove garbage
            os.unlink(diff_html_int)
        else:
            # do not spoil resource with empty files
            os.unlink(diff_output)

        return None, file, has_diff, diff_html
    except Exception as exc:
        eh.log_exception("Exception in _kdiff_thread", exc)
        return str(exc), None, None, None


def _process_to_json(input_path, output_path, lua_to_json):
    fs.make_folder(output_path)
    slow_jobs = []
    for f in os.listdir(input_path):
        full_name = os.path.join(input_path, f)
        if os.path.isfile(full_name):
            if not f.endswith(".struct"):
                slow_jobs.append(
                    [lua_to_json, full_name, os.path.join(output_path, "{}.json".format(f))]
                )
            else:
                shutil.copyfile(full_name, os.path.join(output_path, f))
    return slow_jobs


def _remove_uuid(r, is_shared=False):
    if isinstance(r, dict):
        for k in r.keys():
            if is_shared and k == "uuid":
                del r[k]
            else:
                _remove_uuid(r[k], k.lower() == "shared")
    elif isinstance(r, list):
        for v in r:
            _remove_uuid(v)


def _grab_shared(r, shared_sections):
    if isinstance(r, dict):
        if 'shared' in r:
            t = r['shared']
            if 'uuid' in t and len(t) > 1:
                uuid = t['uuid']
                t.pop('uuid')
                r.pop('shared')
                assert(uuid not in shared_sections)
                shared_sections[uuid] = t
                assert(not r.viewkeys() & t.viewkeys())
                r.update(t)

        for v in r.itervalues():
            _grab_shared(v, shared_sections)
    elif isinstance(r, list):
        for v in r:
            _grab_shared(v, shared_sections)


def _replace_shared(r, shared_sections):
    if isinstance(r, dict):
        if 'shared' in r:
            s = r['shared']
            if s.keys() == ['uuid']:
                uuid = s['uuid']
                r.pop('shared')
                t = shared_sections[uuid]
                assert(not r.viewkeys() & t.viewkeys())
                r.update(t)

        for v in r.itervalues():
            _replace_shared(v, shared_sections)
    elif isinstance(r, list):
        for v in r:
            _replace_shared(v, shared_sections)


def _extract_shared(r):
    shared_sections = {}
    _grab_shared(r, shared_sections)
    _replace_shared(r, shared_sections)


def _remove_backend(r, is_proxy=False):
    if isinstance(r, dict):
        for k in r.keys():
            if is_proxy and (k == 'host' or k == 'port' or k == 'cached_ip'):
                del r[k]
            else:
                _remove_backend(r[k], k.lower() == 'proxy')
    elif isinstance(r, list):
        for v in r:
            _remove_backend(v)


def _write_diff(output_file, content):
    cnt = 0  # Before we woold have another "has_diff" mechanics
    with open(output_file, 'w') as out:
        for r in content:
            out.write(r.encode('utf-8'))
            cnt += 1
    return cnt


def _list_json(json_dir):
    files = os.listdir(json_dir)
    files = [
        file_name for file_name in files
        if ".pickle" not in file_name
    ]
    return set(files)


@rm_notify.notify2()
class CompareBalancerConfigs(bugbanner2.BugBannerTask):
    """
    Exec balancer lua configs and compare results
    """
    class Requirements(sdk2.Requirements):
        disk_space = 20 * 1024  # 20 GiB, 14 detected
        environments = (task_env.TaskRequirements.startrek_client,)
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        old = sdk2.parameters.LastReleasedResource(
            'Balancer old configs',
            resource_type=br.BALANCER_GENCFG_CONFIGS_L7_DIR,
            required=True
        )
        old_yav = sdk2.parameters.LastReleasedResource(
            'Balancer old yav-deploy.conf',
            resource_type=br.BALANCER_L7_YAV_DEPLOY_CONF,
        )
        new = sdk2.parameters.Resource(
            'Balancer new configs',
            resource_type=br.BALANCER_GENCFG_CONFIGS_L7_DIR,
            required=True
        )
        new_yav = sdk2.parameters.Resource(
            'Balancer new yav-deploy.conf',
            resource_type=br.BALANCER_L7_YAV_DEPLOY_CONF,
        )
        lua_to_json = sdk2.parameters.Resource(
            'Prebuilt lua_to_json',
            resource_type=br.LUA_TO_JSON,
            required=True
        )
        remove_uuid = sdk2.parameters.Bool("Remove uuid", default=False)
        remove_backend = sdk2.parameters.Bool("Remove backend", default=False)
        extract_shared = sdk2.parameters.Bool("Extract shared", default=False)

        build_kdiff = sdk2.parameters.Bool("Build kdiff", default=False)
        hash_markers = sdk2.parameters.Bool("Enable hash markers", default=False)
        process_count = sdk2.parameters.Integer("Process count", default=16)

        release_number = sdk2.parameters.Integer('Release number', default=0)

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, BalancerConfigCfg.name)
        self.Context.old_json = BalancerConfigsJson(self, 'Balancer old configs processed to json', 'old_json').id
        self.Context.new_json = BalancerConfigsJson(self, 'Balancer new configs processed to json', 'new_json').id
        self.Context.output = BalancerConfigsDiff(self, 'Balancer configs diffs', 'diffs').id
        self.Context.output_summary = BalancerConfigsDiff(self, 'Balancer configs diff summary', 'diff_summary').id

    def preprocess(self, file, files, json):
        if file not in files:
            return []

        loaded_json = fu.json_load(os.path.join(json, file))
        if self.Parameters.remove_uuid:
            _remove_uuid(loaded_json)
        if self.Parameters.remove_backend:
            _remove_backend(loaded_json)
        if self.Parameters.extract_shared:
            _extract_shared(loaded_json)
        return loaded_json

    def load_json(self, old_files, old_json, new_files, new_json):
        for f in (old_files | new_files):
            try:
                old = self.preprocess(f, old_files, old_json)
                new = self.preprocess(f, new_files, new_json)
                yield f, old, new
            except Exception as exc:
                eh.log_exception("Exception in load_json", exc, task=self)

    def kdiff_simplify(self, file, files, json_configs_path, prefix, output_path):
        """
        Simplify json config to diffable format via kdiff tool
        @param prefix: '.new' or '.old'
        @return: path to simplified json filename
        """
        json_config = os.path.join(json_configs_path, file)
        json_simplified_config = os.path.join(output_path, file + prefix + ".simplified.json")
        logging.info("kdiff_simplify for %s%s outputs to %s", file, prefix, json_simplified_config)
        return json_config, json_simplified_config

    def kdiff_simplify_all(self, old_files, old_json, new_files, new_json, output_path):
        logging.info("kdiff_simplify_all called")
        for file in (old_files | new_files):
            if file not in old_files or file not in new_files:
                logging.debug("File %s not in files, skipped", file)
                continue

            old_input, old_output = self.kdiff_simplify(file, old_files, old_json, ".old", output_path)
            new_input, new_output = self.kdiff_simplify(file, new_files, new_json, ".new", output_path)
            yield {
                "file": file,
                "old_input": old_input,
                "old_output": old_output,
                "new_input": new_input,
                "new_output": new_output,
                "kdiff_uses_argparse": self.kdiff_uses_argparse,
                "hash_markers": self.Parameters.hash_markers,
            }

    def comment_to_st(self, resource_id):
        try:
            if self.Parameters.release_number <= 0:
                logging.debug("No need to send comment")
                return
            try:
                c_info = rmc.get_component(BalancerConfigCfg.name)
                st_helper = STHelper(rm_sec.get_rm_token(self))
                old_major_n, old_minor_n = c_info.get_tag_info_from_build_task(
                    self.Parameters.old.task.id,
                    ARCADIA_URL_KEY,
                )
                new_major_n, new_minor_n = c_info.get_tag_info_from_build_task(
                    self.Parameters.new.task.id,
                    ARCADIA_URL_KEY,
                )
                # FIXME(mvel): use RM SDK instead of formatting
                old_version = "{}-{}".format(old_major_n, old_minor_n)
                new_version = "{}-{}".format(new_major_n, new_minor_n)

                st_helper.comment(
                    release_num=self.Parameters.release_number,
                    text=(
                        "Compare v{new_version} vs v{old_version}:\n"
                        "KDiff results: {task_link}\n"
                        "JsonDiff (legacy): {diff_resource_link}\n"
                    ).format(
                        old_version=old_version,
                        new_version=new_version,
                        task_link=lb.task_wiki_link(self.id),
                        diff_resource_link=lb.resource_wiki_link(resource_id, "results"),
                    ),
                    c_info=c_info,
                )
            except Exception as exc:
                eh.log_exception("Cannot update comment", exc, task=self)
        except Exception as exc:
            eh.log_exception("Cannot get release", exc, task=self)

    def on_execute(self):
        self.add_bugbanner(bugbanner2.Banners.Balancer)

        self.Context.has_diff = False

        old_data = sdk2.ResourceData(self.Parameters.old)
        old_path = str(old_data.path)
        new_data = sdk2.ResourceData(self.Parameters.new)
        new_path = str(new_data.path)
        lua_to_json = str(sdk2.ResourceData(self.Parameters.lua_to_json).path)

        self.kdiff_path = None
        if self.Parameters.build_kdiff:
            sdk2.svn.Arcadia.export(
                "arcadia:/arc/trunk/arcadia/gencfg/custom_generators/balancer_gencfg/kdiff",
                "kdiff",
            )
            self.kdiff_path = "kdiff/simplify.py"
            self.kdiff_uses_argparse = "--input" in fu.read_file(self.kdiff_path)

        old_json = str(sdk2.Resource[self.Context.old_json].path)
        new_json = str(sdk2.Resource[self.Context.new_json].path)

        old_json_jobs = _process_to_json(old_path, old_json, lua_to_json)
        new_json_jobs = _process_to_json(new_path, new_json, lua_to_json)
        all_json_jobs = old_json_jobs + new_json_jobs

        if self.Parameters.process_count > 1:
            logging.info("Started lua_to_json multiprocessing with %s processes", self.Parameters.process_count)
            pool = multiprocessing.Pool(processes=self.Parameters.process_count)
            try:
                pool.imap_unordered(_lua_to_json_thread, all_json_jobs)
            finally:
                pool.close()
                pool.join()
            logging.info("lua_to_json multiprocessing finished")
        else:
            logging.warning("Use single-process (slow) lua_to_json processing")
            map(_lua_to_json_thread, all_json_jobs)

        old_files = _list_json(old_json)
        new_files = _list_json(new_json)

        output_path = str(sdk2.Resource[self.Context.output].path)
        fs.make_folder(output_path)

        json_content = self.load_json(old_files, old_json, new_files, new_json)
        simplified_content = {}
        if self.kdiff_path:
            simplified_content = self.kdiff_simplify_all(
                old_files, old_json, new_files, new_json, output_path
            )

            logging.info("Started main diff multiprocessing with %s processes", self.Parameters.process_count)

            if self.Parameters.process_count > 1:
                pool = multiprocessing.Pool(processes=self.Parameters.process_count)
                try:
                    diffs = list(pool.imap_unordered(_jsondiff_thread, json_content))
                    simplified_diffs = list(pool.imap_unordered(_kdiff_thread, simplified_content))
                finally:
                    pool.close()
                    pool.join()
            else:
                diffs = list(map(_jsondiff_thread, json_content))
                simplified_diffs = list(map(_kdiff_thread, simplified_content))

            logging.info("Main diff multiprocessing finished")
        else:
            logging.warning("Use single-process (slow) processing")
            diffs = map(_jsondiff_thread, json_content)
            simplified_diffs = map(_kdiff_thread, simplified_content)

        for file, diff in diffs:

            if _write_diff(
                os.path.join(output_path, file + '.diff'),
                jsondiff.render_text([(file, diff)], group_items=True)
            ) > 2:
                logging.debug("File '%s' has diff", file)
                # Before we would have another "has_diff" mechanics. See also MINOTAUR-1056
                self.Context.has_diff = True

            _write_diff(
                os.path.join(output_path, file + '.html'),
                jsondiff.render_html([(file, diff)], group_items=True)
            )

        logging.info("=== SIMPLIFIED DIFF RESULTS ===")
        diff_list = []
        for error, file, has_diff, diff_html in simplified_diffs:
            if error:
                eh.check_failed("Simplified diff exception: {}".format(error))

            logging.info("Simplified diff result: %s, has diff: %s, output: %s", file, has_diff, diff_html)
            if has_diff:
                base_diff = os.path.basename(diff_html)
                diff_list.append(base_diff)

        diff_list = sorted(diff_list)
        for base_diff in diff_list:
            self.set_info(u2.resource_redirect_link(self.Context.output, base_diff, base_diff), do_escape=False)

        self.Context.diff_list = diff_list
        if self.kdiff_path:
            # when kdiff mode is enabled, use kdiff semantics (see MINOTAUR-1056 bug)
            self.Context.has_diff = bool(diff_list)

        with open(str(sdk2.ResourceData(self.Parameters.old_yav).path)) as yav_file:
            yav_old = yav_file.readlines()
        with open(str(sdk2.ResourceData(self.Parameters.new_yav).path)) as yav_file:
            yav_new = yav_file.readlines()

        self.Context.yav_diff = '\n'.join(map(str.rstrip, difflib.unified_diff(
            yav_old, yav_new, fromfile="yav-deploy.conf.old", tofile="yav-deploy.conf.new", n=1)))

        output_summary_path = str(sdk2.Resource[self.Context.output_summary].path)
        fs.make_folder(output_summary_path)

        _write_diff(
            os.path.join(output_summary_path, 'summary.diff'),
            jsondiff.render_text(diffs, group_items=True)
        )
        _write_diff(
            os.path.join(output_summary_path, 'summary.html'),
            jsondiff.render_html(diffs, group_items=True)
        )

        self.comment_to_st(self.Context.output_summary)

    @sdk2.header()
    def header(self):
        diff_list = self.Context.diff_list
        yav_diff = self.Context.yav_diff

        if diff_list:
            def name_formatter(config_name):
                return u2.resource_redirect_link(
                    self.Context.output, config_name.replace(".json.simplified.html", ""), config_name
                )

            table = config_grouping.generate_table(diff_list, name_formatter)
        else:
            table = []

        return [{
            "content": {
                "<h3>yav-deploy.conf Diff</h3>": "<pre>{}</pre>".format(yav_diff)
            },
            "helperName": "",
        }, {
            "content": {
                "<h3>Config Diffs</h3>": table
            },
            "helperName": "",
        }]
