# coding: utf-8

_COMMON_DC_SUFFIXES = [
    "_{dc}.cfg",
    "_{dc}_only.cfg",
]

_DC = [
    "sas",
    "man",
    "vla",

    "msk",  # legacy naming, e.g. for yastat
    "unknown",  # also legacy naming, e.g. for yastat
]


def _generate_suffixes():
    common_suffixes = []

    for dc in _DC:
        for suffix in _COMMON_DC_SUFFIXES:
            common_suffixes.append(suffix.format(dc=dc) + ".json.simplified.html")

    for dc in _DC:
        common_suffixes.append("_{dc}_tun.cfg.json.simplified.html".format(dc=dc))

    # should be last one as "common"
    common_suffixes.append(".cfg.json.simplified.html")

    return common_suffixes


_COMMON_SUFFIXES = _generate_suffixes()


def _separate_configs_by_groups(diff_list):
    """
    @param diff_list: list of file names
    """
    diff_groups = {}
    for name in diff_list:
        match_one_of = False
        for suffix in _COMMON_SUFFIXES:
            if not name.endswith(suffix):
                continue

            group_name = name[:-len(suffix)]
            if group_name not in diff_groups:
                diff_groups[group_name] = []
            diff_groups[group_name].append(name)
            match_one_of = True
            break  # do not match more than one suffix

        # push ungrouped configs one by one
        if not match_one_of:
            diff_groups[name] = [name]

        assert match_one_of, "Name {} does not match any of given suffixes".format(name)

    return diff_groups


def _known_unclassified(config_name):
    if "any_test" in config_name:
        return True

    if "yastat_test" in config_name:
        return True

    return False


def generate_table(diff_list, name_formatter):
    diff_groups = _separate_configs_by_groups(diff_list)
    group_names = sorted(diff_groups.keys())
    table_header = [
        {"key": "group_name", "title": "Group"},
        {"key": "production", "title": "Production"},
        {"key": "only", "title": "Geo only"},
    ]
    group_names_column = []
    config_columns = [[] for _ in xrange(4)]

    for group_name in group_names:

        group_names_column.append(group_name)
        for i in xrange(4):
            config_columns[i].append("")

        # factor config names by DC
        names_per_location = {
            dc: [] for dc in _DC
        }
        for config_name in diff_groups[group_name]:
            put = False
            for dc in names_per_location:
                if "_{}".format(dc) in config_name:
                    # already sorted properly
                    names_per_location[dc].append(config_name)
                    put = True

            if not put:
                names_per_location["unknown"].append(config_name)
                if not _known_unclassified(config_name):
                    raise Exception("Config should be put at least in one basket: {}. ".format(config_name))

        for dc in sorted(names_per_location.keys()):
            dc_names = names_per_location[dc]
            if not dc_names:
                continue

            # each DC can have up to 4 configs
            group_names_column.append("")
            for i in xrange(4):
                config_columns[i].append("")

            assert len(dc_names) <= 4

            for config_name in dc_names:
                link = name_formatter(config_name)

                if "_only.cfg.hamster." in config_name:
                    config_columns[3][-1] = link
                elif "_only.cfg." in config_name:
                    config_columns[2][-1] = link
                elif ".cfg.hamster." in config_name:
                    config_columns[1][-1] = link
                else:
                    config_columns[0][-1] = link

            # for i in xrange(4):
            #    print config_columns[i][-1], "\t\t",
            # print

        for i in xrange(4):
            assert(len(config_columns[0]) == len(config_columns[i]))

    table = {
        "header": table_header,
        "body": {
            "group_name": group_names_column,
            "production": config_columns[0],
            "hamster": config_columns[1],
            "only": config_columns[2],
            "hamster_only": config_columns[3],
        }
    }

    return table
