# coding=utf-8
import jinja2

import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.balancer.load.BalancerLoadTest import BalancerLoadTest
from sandbox.projects.balancer.load.common import BalancerLoadCommonParameters1
from sandbox.projects.common.constants.constants import ARCADIA_URL_KEY
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.helpers import startrek_helper
import sandbox.projects.release_machine.core.task_env as task_env


REPORT_TEMPLATE = '''
<table>
<tr>
   <th>Plan</th>
   <th>Old RPS</th>
   <th>New RPS</th>
   <th>Delta</th>
   <th>Sandbox</th>
   <th>Lunapark</th>
</tr>
{% for item in items %}
<tr {{loop.cycle('', 'style="background-color: #f2f2f2;"')}}>
   <td>{{item[0]}}</td>
   <td>{{item[1]}}</td>
   <td>{{item[2]}}</td>
   <td>{{item[3]}}</td>
   <td>{{item[4]}}</td>
   <td>{{item[5]}}</td>
</tr>
{% endfor %}

'''


class BalancerLoadCompare(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]

    class Parameters(BalancerLoadCommonParameters1):
        with sdk2.parameters.RadioGroup('binary_old') as binary_old:
            binary_old.values['url'] = binary_old.Value('url', default=True)
            binary_old.values['sandbox'] = binary_old.Value('sandbox')

        with binary_old.value['url']:
            binary_old_url = sdk2.parameters.String('binary_old_url', required=True)
        with binary_old.value['sandbox']:
            binary_old_sandbox_resource = sdk2.parameters.Resource('binary_old_sandbox_resource (leave empty for last released)')

        configs_old = sdk2.parameters.Resource(
            'configs_old (leave empty for last released)',
            resource_type=balancer_resources.BALANCER_GENCFG_CONFIGS_TEST_DIR,
        )

        with sdk2.parameters.RadioGroup('binary_new') as binary_new:
            binary_new.values['url'] = binary_new.Value('url', default=True)
            binary_new.values['sandbox'] = binary_new.Value('sandbox')

        with binary_new.value['url']:
            binary_new_url = sdk2.parameters.String('binary_new_url', required=True)
        with binary_new.value['sandbox']:
            binary_new_sandbox_resource = sdk2.parameters.Resource('binary_new_sandbox_resource (leave empty for last released)')

        configs_new = sdk2.parameters.Resource(
            'configs_new (leave empty for last released)',
            resource_type=balancer_resources.BALANCER_GENCFG_CONFIGS_TEST_DIR,
        )

        release_component = sdk2.parameters.String('Release component (for release machine)')
        release_number = sdk2.parameters.Integer('Release number (for release machine)')

    def get_last_released_binary(self):
        res = balancer_resources.BALANCER_EXECUTABLE.find(
            attrs={'released': ctt.ReleaseStatus.UNSTABLE}
        ).order(-sdk2.Resource.id).first()
        if not res:
            raise ce.TaskError('Last released binary is not found')
        return res

    def get_last_released_configs(self):
        res = balancer_resources.BALANCER_GENCFG_CONFIGS_TEST_DIR.find(
            attrs={'released': ctt.ReleaseStatus.STABLE}
        ).order(-sdk2.Resource.id).first()
        if not res:
            raise ce.TaskError('Last released configs is not found')
        return res

    def on_enqueue(self):
        if (
            self.Parameters.binary_old == 'sandbox' and not self.Parameters.binary_old_sandbox_resource and not self.Parameters.configs_old
            and self.Parameters.binary_new == 'sandbox' and not self.Parameters.binary_new_sandbox_resource and not self.Parameters.configs_new
        ):
            raise ce.TaskError("Don't want to compare all last released")

        if self.Parameters.binary_old == 'sandbox' and not self.Parameters.binary_old_sandbox_resource:
            self.Parameters.binary_old_sandbox_resource = self.get_last_released_binary()
        if not self.Parameters.configs_old:
            self.Parameters.configs_old = self.get_last_released_configs()

        if self.Parameters.binary_new == 'sandbox' and not self.Parameters.binary_new_sandbox_resource:
            self.Parameters.binary_new_sandbox_resource = self.get_last_released_binary()
        if not self.Parameters.configs_new:
            self.Parameters.configs_new = self.get_last_released_configs()

    def get_results(self):
        old_results = sdk2.Task[self.Context.tasks[0]].Parameters.results
        new_results = sdk2.Task[self.Context.tasks[1]].Parameters.results

        def get_shooting_result(plan):
            if plan in old_results:
                old_rps = old_results[plan]['rps']
                old_task_id = old_results[plan]['task_id']
                old_lunapark_id = old_results[plan].get('lunapark_id')
            else:
                old_rps = 'SKIPPED'
                old_task_id = old_lunapark_id = None

            if plan in new_results:
                new_rps = new_results[plan]['rps']
                new_task_id = new_results[plan]['task_id']
                new_lunapark_id = new_results[plan].get('lunapark_id')
            else:
                new_rps = 'SKIPPED'
                new_task_id = new_lunapark_id = None

            if old_rps in ('ERROR', 'SKIPPED') or new_rps in ('ERROR', 'SKIPPED'):
                delta = 'ERROR'
                bad_delta = True
            elif old_rps == 0 or new_rps == 0:
                delta = '∞'
                bad_delta = True
            else:
                delta = float(new_rps - old_rps) / old_rps
                bad_delta = abs(delta) >= 0.03
                delta = '{:.2f}'.format(delta * 100)

            return plan, old_rps, old_task_id, old_lunapark_id, new_rps, new_task_id, new_lunapark_id, delta, bad_delta

        plans = sorted(set(old_results) | set(new_results))
        return [get_shooting_result(plan) for plan in plans]

    def on_execute(self):
        if not self.Context.tasks:
            tasks = [
                BalancerLoadTest(
                    self,
                    description=self.Parameters.description + ' old',
                    binary=self.Parameters.binary_old,
                    binary_url=self.Parameters.binary_old_url,
                    binary_sandbox_resource=self.Parameters.binary_old_sandbox_resource,
                    configs=self.Parameters.configs_old,
                    target=self.Parameters.target,
                    tank=self.Parameters.tank,
                ).enqueue().id,
                BalancerLoadTest(
                    self,
                    description=self.Parameters.description + ' new',
                    binary=self.Parameters.binary_new,
                    binary_url=self.Parameters.binary_new_url,
                    binary_sandbox_resource=self.Parameters.binary_new_sandbox_resource,
                    configs=self.Parameters.configs_new,
                    target=self.Parameters.target,
                    tank=self.Parameters.tank,
                ).enqueue().id,
            ]

            self.Context.tasks = tasks
            raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            old_results = sdk2.Task[self.Context.tasks[0]].Parameters.results
            new_results = sdk2.Task[self.Context.tasks[1]].Parameters.results

            def get_item(plan, old_rps, old_task_id, old_lunapark_id, new_rps, new_task_id, new_lunapark_id, delta, bad_delta):
                if bad_delta:
                    delta = '<span style="color: red">{}</span>'.format(delta)

                sandbox = []
                if old_task_id:
                    sandbox.append('<a href="https://sandbox.yandex-team.ru/task/{}">Old</a>'.format(old_task_id))
                if new_task_id:
                    sandbox.append('<a href="https://sandbox.yandex-team.ru/task/{}">New</a>'.format(new_task_id))
                sandbox = ' '.join(sandbox)

                if old_lunapark_id and new_lunapark_id:
                    lunapark_link = (
                        '<a href="https://lunapark.yandex-team.ru/compare/#jobs={0},{1}&tab=test_data&helper=all'
                        '&cases=&plotGroup=additional&metricGroup=&target=">{0} vs {1}</a>'
                    ).format(old_lunapark_id, new_lunapark_id)
                else:
                    lunapark_link = ''

                return plan, old_rps, new_rps, delta, sandbox, lunapark_link

            items = (get_item(*shooting_result) for shooting_result in self.get_results())

            self.set_info(jinja2.Environment().from_string(REPORT_TEMPLATE).render(items=items), do_escape=False)

    def send_st_messsage(self):
        if not self.Parameters.release_component or not self.Parameters.release_number:
            return

        c_info = rmc.COMPONENTS[self.Parameters.release_component]()
        def get_version(res):
            if not res:
                return 'unknown'

            major_n, minor_n = c_info.get_tag_info_from_build_task(
                res.task.id,
                ARCADIA_URL_KEY,
            )
            return '{}-{}'.format(major_n, minor_n)

        text = []
        if self.Parameters.release_component == 'balancer':
            text.append('**Compare balancer v{} vs v{}**'.format(
                get_version(self.Parameters.binary_old_sandbox_resource),
                get_version(self.Parameters.binary_new_sandbox_resource),
            ))
        elif self.Parameters.release_component == 'balancer_config':
            text.append('**Compare configs v{} vs v{}**'.format(
                get_version(self.Parameters.configs_old),
                get_version(self.Parameters.configs_new),
            ))

        if not self.Context.tasks:
            text.append('**!!Load tests have not finished. You should see and rerun them.!!**')
        else:
            bad_results = False
            results_text = ['#|']

            def add_table_row(*data):
                results_text.append('||{}||'.format(' | '.join(str(i) for i in data)))

            add_table_row('**Plan**', '**Old RPS**', '**New RPS**', '**Delta**', '**Sandbox**', '**Lunapark**')

            for plan, old_rps, old_task_id, old_lunapark_id, new_rps, new_task_id, new_lunapark_id, delta, bad_delta in self.get_results():
                if bad_delta:
                    delta = '!!{}!!'.format(delta)
                    bad_results = True

                sandbox = []
                if old_task_id:
                    sandbox.append('((https://sandbox.yandex-team.ru/task/{} Old))'.format(old_task_id))
                if new_task_id:
                    sandbox.append('((https://sandbox.yandex-team.ru/task/{} New))'.format(new_task_id))
                sandbox = ' '.join(sandbox)

                if old_lunapark_id and new_lunapark_id:
                    lunapark_link = (
                        '((https://lunapark.yandex-team.ru/compare/#jobs={0},{1}&tab=test_data&helper=all'
                        '&cases=&plotGroup=additional&metricGroup=&target= {0} vs {1}))'
                    ).format(old_lunapark_id, new_lunapark_id)
                else:
                    lunapark_link = ''

                add_table_row(plan, old_rps, new_rps, delta, sandbox, lunapark_link)

            results_text.append('|#')

            if bad_results:
                text.append('**!!Load tests have bad results. Check deltas.!!**')
            else:
                text.append('**Load tests are good.**')
            text += results_text

        text.append('Sent from https://sandbox.yandex-team.ru/task/{}'.format(self.id))

        text = '\n'.join(text)

        rm_token = rm_sec.get_rm_token(self)
        st_helper = startrek_helper.STHelper(rm_token)
        st_helper.comment(self.Parameters.release_number, text, c_info)

    def on_finish(self, prev_status, status):
        self.send_st_messsage()
