import collections
import datetime
import json
import logging
import numpy
import os
import tarfile
import time

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.projects.balancer.load.BalancerLoadShootViaTankapi import BalancerLoadShootViaTankapi
from sandbox.projects.balancer.load.common import request, BalancerLoadCommonParameters2
from sandbox.sdk2 import yav


PANDORA_TEMPLATE = '''pandora:
  enabled: true
  config_content:
    pools:
      - id: pool
        gun:
          type: {gun_type}
          ssl: {ssl}
          target: '{target_host}:{target_port}'
        ammo:
          type: uri
          file: ./ammo
        result:
          type: phout
          destination: ./phout.log
        rps: {shooting_plan}
        startup:
          type: once
          times: 500
phantom:
  enabled: false
uploader:
  operator: robot-rainbow-dash
  task: '{st_task}'
  ignore_target_lock: true
'''

TARGETS = {
    'balancer_load_target': 'man2-1693-dc6-man-web-nmeta-tar-acc-7000.gencfg-c.yandex.net',
    'balancer_load_target_testing': 'man0-0467-e0a-man-web-nmeta-tar-de7-7000.gencfg-c.yandex.net',
}

TANKS = {
    'tank1': 'man2-6390-3cc-man-web-nmeta-tan-b09-7400.gencfg-c.yandex.net:7400',
    'tank2': 'man1-3975-man-web-nmeta-tank-balancer-7400.gencfg-c.yandex.net:7400',
}


def get_backends_rewrite(name):
    if name == 'disabled':
        return None

    base_dir = os.path.dirname(os.path.abspath(__file__))
    return open(os.path.join(base_dir, 'backends_{}.json'.format(name))).read()


def get_sd_config(name):
    if name == 'disabled':
        return None

    base_dir = os.path.dirname(os.path.abspath(__file__))
    return open(os.path.join(base_dir, 'sd_{}.json'.format(name))).read()


def update_nanny_service_file(runtime_attrs, name, file_type, spec):
    resources = runtime_attrs['resources']
    file_type_full = file_type + '_files'

    replaced = False

    for file_type_current in ['sandbox_files', 'url_files', 'static_files']:
        i = 0
        while i < len(resources[file_type_current]):
            if resources[file_type_current][i]['local_path'] == name:
                if file_type_current == file_type_full and not replaced:
                    resources[file_type_current][i] = spec
                    resources[file_type_current][i]['local_path'] = name
                    replaced = True
                else:
                    resources[file_type_current] = resources[file_type_current][:i] + resources[file_type_current][i + 1:]
                    continue
            i += 1

    if not replaced:
        resources[file_type_full].append(spec)
        resources[file_type_full][-1]['local_path'] = name


def update_nanny_service_resousre(runtime_attrs, name, resource):
    if type(resource) == int:
        file_type = 'sandbox'
        res = sdk2.Resource[resource]
        spec = {
            'resource_id': str(resource),
            'resource_type': str(res.type),
            'task_id': str(res.task.id),
            'task_type': str(res.task.type),
        }
    else:
        file_type = 'url'
        spec = {
            'url': resource,
            'extract_path': '',
        }

    update_nanny_service_file(runtime_attrs, name, file_type, spec)


def get_results(phout):
    rps = collections.defaultdict(int)
    quant = collections.defaultdict(list)

    start_time = None
    end_time = None

    for line in phout:
        line = line.strip().split('\t', 11)
        cur_time = int(float(line[0]))
        rtt = int(line[2])
        http_status_code = int(line[11])

        if http_status_code != 200:
            continue

        start_time = min(start_time, cur_time) if start_time is not None else cur_time
        end_time = max(end_time, cur_time) if end_time is not None else cur_time

        rps[cur_time] += 1
        quant[cur_time].append(rtt)

    assert start_time is not None
    end_time += 1

    def get_rps(start, end):
        r = 0
        for i in xrange(start, end):
            r += rps[i]
        return r / (end - start)

    def get_percentiles(start, end, percentiles):
        a = []
        for i in xrange(start, end):
            a += quant[i]
        a = numpy.array(a)
        return {p: numpy.percentile(a, p) / 1000 for p in percentiles}

    max_rps = get_rps(end_time - 35, end_time - 5)
    percentiles = get_percentiles(start_time + 5, start_time + 35, (50, 90, 95, 99))

    return start_time, end_time, max_rps, percentiles


class BalancerLoadShoot(sdk2.Task):
    class Parameters(BalancerLoadCommonParameters2):
        with sdk2.parameters.RadioGroup('config') as config:
            config.values['url'] = config.Value('url', default=True)
            config.values['sandbox'] = config.Value('sandbox')

        with config.value['url']:
            config_url = sdk2.parameters.String('config_url', required=True)
        with config.value['sandbox']:
            config_sandbox_resource = sdk2.parameters.Resource('config_sandbox_resource', required=True)

        with sdk2.parameters.RadioGroup('backends_rewrite') as backends_rewrite:
            backends_rewrite.values['disabled'] = backends_rewrite.Value('Disabled', default=True)
            backends_rewrite.values['l7heavy'] = backends_rewrite.Value('l7heavy')
            backends_rewrite.values['l7heavy_5xx'] = backends_rewrite.Value('l7heavy_5xx')

        shooting_plan = sdk2.parameters.String('shooting_plan', default='const(100,10m)', required=True)
        shooting_url = sdk2.parameters.String('shooting_url')

        with sdk2.parameters.RadioGroup('shooting_mode') as shooting_mode:
            shooting_mode.values['ka'] = shooting_mode.Value('ka', default=True)
            shooting_mode.values['noka'] = shooting_mode.Value('noka')
            shooting_mode.values['ssl_ka'] = shooting_mode.Value('ssl_ka')
            shooting_mode.values['ssl_noka'] = shooting_mode.Value('ssl_noka')
            shooting_mode.values['http2'] = shooting_mode.Value('http2')

        sandbox_task = sdk2.parameters.String('sandbox_task')

        with sdk2.parameters.Output:
            lunapark_id = sdk2.parameters.Integer('lunapark_id')
            target_rps = sdk2.parameters.Integer('target_rps')
            target_latency = sdk2.parameters.Float('target_latency')

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(
                    name='BalancerLoadShoot_' + self.Parameters.target,
                    capacity=1
                )
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
        )
        return super(BalancerLoadShoot, self).on_enqueue()

    def wait_nanny_service(self, service, target_id):
        for _ in xrange(90):
            logging.debug('Wait snapshot {}'.format(target_id))

            snapshots = request(
                'https://nanny.yandex-team.ru/v2/services/{}/current_state/'.format(service),
                headers={'Authorization': 'OAuth {}'.format(self.nanny_oauth_token)}
            ).json()['content']['active_snapshots']

            for snapshot in snapshots:
                if snapshot['snapshot_id'] == target_id and snapshot['state'] == 'ACTIVE':
                    return

            time.sleep(10)
        else:
            raise Exception('Activate nanny service timeout')

    def update_nanny_service(self, service, balancer, config, rewrite, sd_config):
        logging.debug('Update service {}'.format(service))
        descr = 'Update from https://sandbox.yandex-team.ru/task/{}'.format(self.id)

        runtime_attrs = request(
            'https://nanny.yandex-team.ru/v2/services/{}/runtime_attrs/'.format(service),
            headers={'Authorization': 'OAuth {}'.format(self.nanny_oauth_token)}
        ).json()

        update_nanny_service_resousre(runtime_attrs['content'], 'balancer', balancer)
        update_nanny_service_resousre(runtime_attrs['content'], 'balancer.cfg', config)
        update_nanny_service_file(runtime_attrs['content'], 'zerodiff', 'static', {'content': descr})

        if rewrite is not None:
            update_nanny_service_file(runtime_attrs['content'], 'rewrite.json', 'static', {'content': rewrite})

        if sd_config is not None:
            update_nanny_service_file(runtime_attrs['content'], 'sd_mock_conf.json', 'static', {'content': sd_config})

        update_data = {
            'content': runtime_attrs['content'],
            'snapshot_id': runtime_attrs['_id'],
            'comment': descr,
        }
        target_id = request(
            'https://nanny.yandex-team.ru/v2/services/{}/runtime_attrs/'.format(service),
            method='PUT', data=json.dumps(update_data), headers={'Authorization': 'OAuth {}'.format(self.nanny_oauth_token)}
        ).json()['_id']

        self.set_info(
            'Snapshot <a href="{0}">{0}</a>'.format(
                'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/runtime_attrs_history/{}/'.format(service, target_id)
            ),
            do_escape=False,
        )

        self.wait_nanny_service(service, target_id)
        logging.debug('Active snapshot {}'.format(target_id))

    def shoot_via_lunapark(self):
        ssl = self.Parameters.shooting_mode in ('ssl_ka', 'ssl_noka', 'http2')

        config = PANDORA_TEMPLATE.format(
            gun_type='http2' if self.Parameters.shooting_mode == 'http2' else 'http',
            ssl=ssl,
            target_host=TARGETS[self.Parameters.target],
            target_port=7000 + ssl,
            shooting_plan=self.Parameters.shooting_plan,
            st_task=self.Parameters.sandbox_task or 'BALANCER-397'
        )

        ammo = '[host:yandex.ru]\n'
        if self.Parameters.shooting_mode in ('noka', 'ssl_noka'):
            ammo += '[connection: close]\n'
        ammo += '{}\n'.format(self.Parameters.shooting_url or '/')

        shoot_task = BalancerLoadShootViaTankapi(
            self,
            tank=TANKS[self.Parameters.tank],
            config=config,
            ammo=ammo,
        ).enqueue().id

        logging.debug('Shoot task {} has started'.format(shoot_task))

        self.Context.shoot_task = shoot_task
        raise sdk2.WaitTask([shoot_task], (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH), wait_all=True)

    def on_execute(self):
        if not self.Context.shoot_task:
            self.nanny_oauth_token = yav.Secret('sec-01e5qv3dy1ez09eg787t7ky5ad').data()['oauth']

            self.update_nanny_service(
                self.Parameters.target,
                self.Parameters.binary_url if self.Parameters.binary == 'url' else self.Parameters.binary_sandbox_resource.id,
                self.Parameters.config_url if self.Parameters.config == 'url' else self.Parameters.config_sandbox_resource.id,
                get_backends_rewrite(self.Parameters.backends_rewrite),
                get_sd_config(self.Parameters.backends_rewrite),
            )
            self.shoot_via_lunapark()
        else:
            report = []

            archive_resource = tank_resources.YANDEX_TANK_LOGS.find(
                task=sdk2.Task[self.Context.shoot_task],
            ).first()
            assert archive_resource
            archive_path = str(sdk2.ResourceData(archive_resource).path)

            with tarfile.open(archive_path, 'r:gz') as archive:
                phout = archive.extractfile('phout.log')
                assert phout
                start_time, end_time, max_rps, percentiles = get_results(phout)

            lunapark_id = sdk2.Task[self.Context.shoot_task].Parameters.lunapark_id
            if lunapark_id:
                self.Parameters.lunapark_id = lunapark_id
                report.append('Lunapark link: <a href="https://lunapark.yandex-team.ru/{0}">{0}</a>'.format(lunapark_id))

            def time_to_str(t):
                return datetime.datetime.fromtimestamp(t).strftime('%Y-%m-%d %H:%M:%S')

            yasm_panel_link = (
                'https://yasm.yandex-team.ru/template/panel/balancer_load/service={}/?from={}&to={}'
            ).format(self.Parameters.target, (start_time - 20) * 1000, (end_time + 60) * 1000)

            report += [
                'Start time: {}'.format(time_to_str(start_time)),
                'End time:   {}'.format(time_to_str(end_time)),
                '<a href="{}">Yasm panel link</a>'.format(yasm_panel_link),
                'Max rps: {}'.format(max_rps),
            ]

            for k in sorted(percentiles.keys()):
                report.append('Percentile {}: {} ms'.format(k, percentiles[k]))
            self.set_info('<br>'.join(report), do_escape=False)

            assert max_rps > 0
            assert percentiles[95] > 0

            self.Parameters.target_rps = max_rps
            self.Parameters.target_latency = percentiles[95]
