import jinja2

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.balancer.load.BalancerLoadShoot import BalancerLoadShoot
from sandbox.projects.balancer.load.common import BalancerLoadCommonParameters2


REPORT_TEMPLATE = '''
<table style="font-size: 15px;">
<tr>
   <th>Plan</th>
   <th>Max RPS</th>
   <th>Latency q95</th>
   <th>Sandbox task</th>
   <th>Lunapark job</th>
</tr>
{% for item in items %}
<tr {{loop.cycle('', 'style="background-color: #f2f2f2;"')}}>
   <td>{{item[0]}}</td>
   <td>{{item[1]}}</td>
   <td>{{item[2]}}</td>
   <td><a href="https://sandbox.yandex-team.ru/task/{{item[3]}}">{{item[3]}}</a></td>
   <td>{{item[4]}}</td>
</tr>
{% endfor %}

'''


class Plan():
    def __init__(self, name, plan, config_name, mode='ka', backends_rewrite='disabled', url='/'):
        self.name = name
        self.plan = plan
        self.config_name = config_name
        self.mode = mode
        self.backends_rewrite = backends_rewrite
        self.url = url


def default_plan(start, end=None):
    if end is None:
        end = start + 4000
    return (
        '[{{type: const, ops: {start}, duration: 1m}}, '
        '{{type: line, from: {start}, to: {end}, duration: 10m}}]'
    ).format(start=start, end=end)


plans = [
    Plan(
        'term_2workers_ssl_ka', default_plan(5000), 'l7heavy_production_tun_man',
        backends_rewrite='l7heavy',
        mode='ssl_ka',
    ),
    Plan(
        'term_5xx_2workers_ssl_ka', default_plan(2000), 'l7heavy_production_tun_man',
        backends_rewrite='l7heavy_5xx',
        mode='ssl_ka',
    ),
    Plan(
        'term_2workers_ssl_noka', default_plan(500, 3000), 'l7heavy_production_tun_man',
        backends_rewrite='l7heavy',
        mode='ssl_noka',
    ),
    Plan(
        'term_2workers_http2', default_plan(4000), 'l7heavy_production_tun_man',
        backends_rewrite='l7heavy',
        mode='http2',
    ),
    Plan(
        'knoss_search_2workers_ka', default_plan(3500), 'l7heavy_service_search_man',
        backends_rewrite='l7heavy',
        url='/jsonproxy',
    ),
    Plan(
        'knoss_search_2workers_5xx_ka', default_plan(1000), 'l7heavy_service_search_man',
        backends_rewrite='l7heavy_5xx',
        url='/jsonproxy',
    ),
    Plan(
        'knoss_search_2workers_noka', default_plan(3000), 'l7heavy_service_search_man',
        backends_rewrite='l7heavy',
        url='/jsonproxy',
        mode='noka',
    ),
]


class BalancerLoadTest(sdk2.Task):
    class Parameters(BalancerLoadCommonParameters2):
        configs = sdk2.parameters.Resource(
            'configs',
            resource_type=balancer_resources.BALANCER_GENCFG_CONFIGS_TEST_DIR,
            required=True,
        )

        with sdk2.parameters.Output:
            results = sdk2.parameters.JSON('results')

    def on_execute(self):
        if not self.Context.tasks:
            configs_id = self.Parameters.configs.id

            tasks = []

            for plan in plans:
                tasks.append((plan.name, BalancerLoadShoot(
                    self,
                    description='{} {}'.format(self.Parameters.description, plan.name),
                    binary=self.Parameters.binary,
                    binary_url=self.Parameters.binary_url,
                    binary_sandbox_resource=self.Parameters.binary_sandbox_resource,
                    target=self.Parameters.target,
                    tank=self.Parameters.tank,
                    config='url',
                    config_url='https://proxy.sandbox.yandex-team.ru/{}/{}.cfg'.format(configs_id, plan.config_name),
                    backends_rewrite=plan.backends_rewrite,
                    shooting_plan=plan.plan,
                    shooting_url=plan.url,
                    shooting_mode=plan.mode,
                ).enqueue().id))

            if not tasks:
                return
            self.Context.tasks = tasks

            task_ids = [task_id for _, task_id in self.Context.tasks]
            raise sdk2.WaitTask(task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            results = {}

            for plan, task_id in self.Context.tasks:
                task = sdk2.Task[task_id]

                plan_result = {'task_id': task_id}

                if task.Parameters.lunapark_id:
                    plan_result['lunapark_id'] = task.Parameters.lunapark_id

                if task.status in ctt.Status.Group.SUCCEED:
                    plan_result['rps'] = task.Parameters.target_rps
                    plan_result['latency'] = task.Parameters.target_latency
                else:
                    plan_result['rps'] = plan_result['latency'] = 'ERROR'

                results[plan] = plan_result

            self.Parameters.results = results

    @sdk2.header()
    def report(self):
        if not self.Context.tasks:
            return

        def get_item(plan, task_id):
            task = sdk2.Task[task_id]

            if task.status in ctt.Status.Group.SUCCEED:
                target_rps = task.Parameters.target_rps
                target_latency = task.Parameters.target_latency
            elif task.status in ctt.Status.Group.BREAK:
                target_rps = target_latency = 'ERROR'
            else:
                target_rps = target_latency = 'In progress'

            if task.Parameters.lunapark_id:
                lunapark_link = '<a href="https://lunapark.yandex-team.ru/{0}">{0}</a>'.format(task.Parameters.lunapark_id)
            else:
                lunapark_link = ''

            return (
                plan,
                target_rps,
                target_latency,
                task_id,
                lunapark_link,
            )

        items = (get_item(plan, task_id) for plan, task_id in self.Context.tasks)
        return jinja2.Environment().from_string(REPORT_TEMPLATE).render(items=items)
