from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common.build import parameters as ya_make_parameters

import datetime as dt
import re


ServiceDescs = {
    'production': {
        'sandbox_release_type': 'stable',
        'bass_release_type': 'production',
    },
    'priemka': {
        'sandbox_release_type': 'prestable',
        'bass_release_type': 'production',
    },
}


class ReleaseBassBeta(sdk2.Task):
    """ Release BASS beta version (under construction) """
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        use_existing_task = sdk2.parameters.Bool('Use existing BUILD_BASS task')

        with use_existing_task.value[True]:
            subtask = sdk2.parameters.Task("BUILD_BASS task")

        with use_existing_task.value[False]:
            with sdk2.parameters.Group('Create new release') as create_new_release_block:
                arcadia_svn_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)

                with sdk2.parameters.String('Release to', multiline=True) as release_to:
                    release_to.PRIEMKA = release_to.Value('Priemka', default=True)
                    release_to.PRODUCTION = release_to.Value('Production', default=True)

    def on_execute(self):
        if not self.Parameters.use_existing_task:
            build_bass = sdk2.Task['BUILD_BASS']

            self.Parameters.subtask = build_bass(
                self,
                description=self._make_release_name(),
                input_parameters={
                    ya_make_parameters.ArcadiaUrl.name: self.Parameters.arcadia_svn_url,
                    ya_make_parameters.ClearBuild.name: False,
                    ya_make_parameters.StripBinaries.name: False,
                    ya_make_parameters.CheckReturnCode.name: True,
                    'release_type': 'production',  # TODO: Reuse constants from BUILD_BASS
                }
            ).enqueue()
            raise sdk2.WaitTask(self.Parameters.subtask, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=False)

        subtask = self.Parameters.subtask
        if subtask.status not in (ctt.SUCCESS, ctt.RELEASING, ctt.RELEASED):
            raise sdk2.TaskFailure('Subtask {} is in status {}'.format(subtask.id, subtask.status))

        expected_type = ServiceDescs[self.Parameters.release_to]['bass_release_type']
        actual_type = subtask.Context['release_type']
        if actual_type != expected_type:
            raise sdk2.TaskFailure('Please select BUILD_BASS subtask with "release_type"=={}'.format(expected_type))

        self.server.release(
            task_id=int(self.Parameters.subtask),
            type=ServiceDescs[self.Parameters.release_to]['sandbox_release_type'],
            subject=self.Parameters.release_name)

    def _make_release_name(self):
        date = dt.datetime.now().strftime('%Y-%m-%d')
        branch_rev = re.search('^' + re.escape('arcadia:/arc/') + '(.*)$', self.Parameters.arcadia_svn_url)
        return '{}@{}'.format(date, branch_rev)
