# coding: utf-8
import logging
import os
import yaml
import requests
from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.bitbucket.common.git_task_parameters import GitParameters
import sandbox.projects.bitbucket.common.git_helpers as git
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.common.ya_deploy import release_integration

GIT_DIR = 'docker'
TOOLBOX_SCRIPT_FILE = 'toolbox/__main__.py'
DOCKER_REGISTRY_URL = 'registry.yandex.net'
DOCKERFILE_RESOURCE_NAME = 'dockerfile_resource'
NOT_PROCESSED_STATUS = 'Not processed'


class DockerImage:
    def __init__(self, url, version, build_path):
        self.name = url.lstrip('{}/'.format(DOCKER_REGISTRY_URL))
        self.version = version
        self.url = url
        self.build_path = build_path
        self.docker_status = NOT_PROCESSED_STATUS
        self.deploy_status = NOT_PROCESSED_STATUS
        self.sub_task_id = None

    def get_docker_status_line(self):
        return '{}:{} - {}'.format(self.name, self.version, self.docker_status)

    def get_deploy_status_line(self):
        return '{}:{} - {}'.format(self.name, self.version, self.deploy_status)


class BiMstrDockerImageSourceResource(sdk2.Resource):
    """ Dockerfile source directory resource """


class BiMstrUpdateDockerFromGit(sdk2.Task, release_integration.ReleaseToYaDeployTask2):
    """ Builds Dockerfile with script from GIT repository and uploads image to Docker registry """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 * 1024
        environments = [
            PipEnvironment('coloredlogs'),
            PipEnvironment('PyYAML'),
            PipEnvironment('Jinja2'),
            PipEnvironment('requests'),
        ]
        dns = ctm.DnsType.DNS64

    class Parameters(GitParameters):
        with sdk2.parameters.Group('Docker parameters') as docker_group:
            push_to_docker_registry = sdk2.parameters.Bool(
                'Build images and push to Docker registry', default=True, required=True)
            docker_robot_login = sdk2.parameters.String(
                'Robot to push to Docker registry', default='robot-bi-cloud', required=True)
            docker_robot_token_vault = sdk2.parameters.String(
                'Vault item with robot token', default='docker_token', required=True)
        with sdk2.parameters.Group('Yandex.Deploy parameters') as deploy_group:
            release_to_ya_deploy = release_integration.ReleaseToYaDeployParameter2()
            release_type = release_integration.ReleaseTypeParameter2()
            yp_token_vault = release_integration.YpTokenVaultParameter2()

    def checkout_git(self):
        vault_secret_owner = self.Parameters.vault_secret_owner
        if vault_secret_owner == '':
            vault_secret_owner = self.owner
        repo_url = git.get_repo_url_with_credentials(self.Parameters.repo_url,
                                                     self.Parameters.vault_secret_user,
                                                     sdk2.Vault.data(vault_secret_owner,
                                                                     self.Parameters.vault_secret_name))
        logging.info('Loading data from GIT')
        git.git_init(self, GIT_DIR)
        git.git_fetch(self, GIT_DIR, repo_url, self.Parameters.ref_sha or self.Parameters.ref_id, depth=1)
        git.git_checkout(self, GIT_DIR, 'FETCH_HEAD')

    def get_image_list(self):
        image_list = []
        with open('{}/config.yaml'.format(GIT_DIR)) as f:
            config_images = yaml.load(f)['images']
        for image_url in config_images.keys():
            image_list.append(DockerImage(
                image_url,
                config_images[image_url]['version'],
                config_images[image_url]['build_context_path']
            ))
        return image_list

    def create_path_resource(self, resource_name, resource_path):
        resource = BiMstrDockerImageSourceResource(self, resource_name, resource_path)
        sdk2.ResourceData(resource).ready()
        return resource

    def change_dir_to_toolbox(self):
        self.Context.toolbox_script_file_full_path = '{}/{}'.format(GIT_DIR, TOOLBOX_SCRIPT_FILE)
        toolbox_path = os.path.dirname(os.path.realpath(self.Context.toolbox_script_file_full_path))
        logging.info('Changing working directory to {}'.format(toolbox_path))
        os.chdir(toolbox_path)

    def prepare_dockerfile_resource(self, image):
        logging.info('Creating Dockerfile')
        dockerfile_full_path = os.path.realpath(image.build_path)
        if not os.path.isdir(dockerfile_full_path):
            image.docker_status = 'Build path not found'
            return None

        dockerfile_full_name = '{}/Dockerfile'.format(dockerfile_full_path)
        with sdk2.helpers.ProcessLog(self, logger='create_dockerfile') as pl:
            sp.Popen('python {} create_dockerfile --image_name {}'.format(
                os.path.basename(self.Context.toolbox_script_file_full_path),
                image.url
            ), shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()
        if not os.path.isfile(dockerfile_full_name):
            image.docker_status = 'Dockerfile not created'
            return None

        return self.create_path_resource(image.build_path, os.path.dirname(dockerfile_full_name))

    def image_tag_exists(self, image):
        logging.info('Checking image {} with tag {} at docker registry'.format(image.name, image.version))
        session = requests.Session()
        session.headers.update(
            {'AUTHORIZATION': 'OAuth ' + sdk2.Vault.data(self.author, self.Parameters.docker_robot_token_vault)})
        tag_url = 'https://{}/v2/{}/tags/list'.format(DOCKER_REGISTRY_URL, image.name)
        logging.info('Using docker API URL {}'.format(tag_url))
        docker_response = session.get(tag_url).json()
        logging.info('Docker registry response: {}'.format(str(docker_response)))
        if 'tags' not in docker_response.keys():
            logging.info('No tag key returned, assuming no tags exist')
            return False
        tags = set(docker_response['tags'])
        return image.version in tags

    def create_build_image_task(self, resource_id, image):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource_id,
            BuildDockerImageV6.RegistryTags.name: ['{}:{}'.format(image.name, image.version)],
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.docker_robot_login,
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.docker_robot_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.docker_robot_token_vault
        }
        sub_task = task_class(
            task_class.current,
            description='Building and pushing Docker image {}:{} (child of {})'.format(
                image.name, image.version, self.id),
            author=self.Parameters.docker_robot_login,
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        return sub_task

    def create_ya_deploy_release(self, image, release_type):
        image_dict = {
            "name": image.name,
            "tag": image.version,
            "digest": release_integration.EMPTY_DOCKER_HASH_PLACEHOLDER,
            "registry_host": release_integration.REGISTRY_HOST
        }
        release_integration.create_docker_release_inner(
            task=self,
            images=[image_dict],
            release_type=release_type,
            release_author=self.author,
            title='{}:{}'.format(image.name, image.version),
            description=''
        )

    def on_execute(self):
        self.checkout_git()
        images = self.get_image_list()
        self.change_dir_to_toolbox()
        with self.memoize_stage.first_step:
            self.Context.subtasks = []
            if self.Parameters.push_to_docker_registry:
                for image in images:
                    if self.image_tag_exists(image):
                        image.docker_status = 'Already exists in registry'
                    else:
                        resource = self.prepare_dockerfile_resource(image)
                        if resource:
                            sub_task = self.create_build_image_task(resource.id, image)
                            self.Context.subtasks.append(sub_task.id)
                            image.sub_task_id = sub_task.id
                            image.docker_status = 'Docker subtask #{} created'.format(sub_task.id)

                docker_info = '\n'.join(image.get_docker_status_line() for image in images)
                self.set_info('Image Docker status:\n{}'.format(docker_info))

                if self.Context.subtasks:
                    raise sdk2.WaitTask(
                        self.Context.subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.second_step:
            result_errors = []
            for subtask_id in self.Context.subtasks:
                try:
                    subtask = self.find(id=subtask_id).first()
                    subtask_image = subtask.Context.registry_tags[0]
                    if subtask.status in ctt.Status.Group.SUCCEED:
                        self.set_info('{} - Pushed to registry'.format(subtask_image))
                    else:
                        self.set_info('{} - Docker subtask #{} failed'.format(subtask_image, subtask_id))
                except:
                    result_error = 'Could not get status of Docker subtask #{}'.format(subtask_id)
                    result_errors.append(result_error)
                    self.set_info(result_error)
            if self.Parameters.release_to_ya_deploy:
                for image in images:
                    if self.image_tag_exists(image):
                        self.create_ya_deploy_release(image, self.Parameters.release_type)
                        image.deploy_status = 'Notification sent to Deploy'
                    else:
                        image.deploy_status = 'Image version not found in registry'
                        result_errors.append(image.get_deploy_status_line())

                deploy_info = '\n'.join(image.get_deploy_status_line() for image in images)
                self.set_info('Image Deploy status:\n{}'.format(deploy_info))

            if result_errors:
                raise ce.TaskFailure('; '.join(result_errors))
