# -*- coding: utf-8 -*-
import operator
import os
import time

from sandbox import sdk2


def _last_moderation_reducer(reduce_key, rows):
    result_row = max(rows, key=lambda row: int(row['ModerationId']))
    del result_row['ModerationId']
    yield result_row


def _get_filter_automoderator_markups_mapper(prestable_release_time):
    def _filter_automoderator_markups_mapper(row):
        if row['Timestamp'] >= prestable_release_time and row['LogType'] == 'ELT_ROUTE' and row['LogSubType'] == 5 and row['Circuit'] == 'EMC_BIGMOD':
            result_row = {}
            for field in ['ObjectType', 'SubType', 'ObjectId', 'VersionId', 'Markup', 'CampaignId', 'ClientId', 'ModerationId']:
                result_row[field] = row[field]
            yield result_row
    return _filter_automoderator_markups_mapper


class BigmodAutomoderatorB2B(sdk2.Task):

    name = 'BIGMOD_AUTOMODERATOR_B2B'

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('B2B parameters') as b2b_parameters:
            yt_cluster = sdk2.parameters.String('YT cluster', default='hahn')
            b2b_dir = sdk2.parameters.String('B2B directory', default='//home/modadvert/test/bigmod_b2b')
            prestable_logs = sdk2.parameters.String('Prestable logs', default='//logs/bigmod-prestable-common-logs/30min')
            production_logs = sdk2.parameters.String('Production logs', default='//logs/bigmod-production-common-logs/30min')
            log_tables_num = sdk2.parameters.Integer('Log tables to compare number', default=3)
            prestable_release_time = sdk2.parameters.Integer('Presteble release time (milliseconds)', required=True)

        with sdk2.parameters.Output():
            st_comment = sdk2.parameters.String("Startrek comment")

    def _collect_logs(self):
        import yt.wrapper

        with yt.wrapper.OperationsTracker(print_progress=True) as yt_tracker:
            self.Context.prestable_tables = sorted(self.yt_client.list(self.Parameters.prestable_logs, absolute=True))[-self.Parameters.log_tables_num:]
            self.Context.production_tables = sorted(self.yt_client.list(self.Parameters.production_logs, absolute=True))[-self.Parameters.log_tables_num:]
            yt_tracker.add(
                self.yt_client.run_map_reduce(
                    _get_filter_automoderator_markups_mapper(self.Parameters.prestable_release_time),
                    _last_moderation_reducer,
                    source_table=self.Context.prestable_tables,
                    destination_table=os.path.join(self.Context.excecution_dir, 'prestable'),
                    reduce_by=['ObjectType', 'ObjectId', 'VersionId']
                )
            )
            yt_tracker.add(
                self.yt_client.run_map_reduce(
                    _get_filter_automoderator_markups_mapper(self.Parameters.prestable_release_time),
                    _last_moderation_reducer,
                    source_table=self.Context.production_tables,
                    destination_table=os.path.join(self.Context.excecution_dir, 'production'),
                    reduce_by=['ObjectType', 'ObjectId', 'VersionId']
                )
            )

    def _get_b2b_messages(self):
        from sandbox.projects.bigmod.b2b_helpers.stat import get_stat_table

        diff_verdicts_url = u'https://yt.yandex-team.ru/{cluster}/navigation?path={diff_path}'.format(
            cluster=self.Parameters.yt_cluster,
            diff_path=os.path.join(self.Context.excecution_dir, 'comparison', 'diff_markups')
        )
        message_chunks = [u"Таблица с объектами, на которых не совпали ModerationMood'ы: {}\n".format(diff_verdicts_url)]

        for description, dir_name in [
            (u'версиям', 'versions'),
            (u'кампаниям', 'campaign'),
            (u'клиентам', 'client')
        ]:
            global_counts_path = os.path.join(self.Context.excecution_dir, 'comparison', 'global', dir_name, 'count')
            global_counts_rows = sorted(self.yt_client.read_table(global_counts_path), key=operator.itemgetter('ObjectType', 'SubType'))
            global_message = u'В разбивке по типам:\n{}'.format(get_stat_table('mood', ['ObjectType', 'SubType'], global_counts_rows))

            rule_counts_path = os.path.join(self.Context.excecution_dir, 'comparison', 'rule', dir_name, 'count')
            rule_counts_rows= sorted(self.yt_client.read_table(rule_counts_path), key=operator.itemgetter('ObjectType', 'SubType', 'rule'))
            rule_diff_table = get_stat_table('mood', ['ObjectType', 'SubType', 'rule'], rule_counts_rows, hide_tolerance=0.0)
            rule_message = ''
            if rule_diff_table:
                rule_message += u'Фильтры с изменившимися вердиктами:\n{}\n'.format(rule_diff_table)

            message_chunks.append(u'<{{Статистика по {description}\n{global_message}\n{rule_message}}}>'.format(
                description=description,
                global_message=global_message,
                rule_message=rule_message
            ))

        flags_counts_path = os.path.join(self.Context.excecution_dir, 'comparison', 'flags', 'count')
        flags_counts_rows = sorted(self.yt_client.read_table(flags_counts_path), key=operator.itemgetter('ObjectType', 'SubType', 'flag'))
        flags_message = u'В разбивке по типам:\n{}'.format(get_stat_table('flag', ['ObjectType', 'SubType', 'flag'], flags_counts_rows, hide_tolerance=0.0))

        message_chunks.append(u'<{{Статистика по флагам\n{flags_message}}}>'.format(
            flags_message=flags_message
        ))
        return '\n'.join(message_chunks)

    def on_execute(self):
        from sandbox.projects.bigmod.b2b_helpers.automoderator_comparison import AutomoderatorComparison

        yt_token = sdk2.yav.Secret('sec-01e2yzfnsfecqefkq7c2n2hzdx').data()['nirvana-secret']
        self.Context.excecution_dir = os.path.join(self.Parameters.b2b_dir, time.strftime("%Y-%m-%dT%H:%M", time.localtime()))
        comparison = AutomoderatorComparison(self.Parameters.yt_cluster, yt_token, self.Context.excecution_dir)
        self.yt_client = comparison.yt_client
        self.yt_client.create(type='map_node', path=self.Context.excecution_dir, recursive=True, ignore_existing=True)
        self._collect_logs()
        comparison.run()
        self.Parameters.st_comment = self._get_b2b_messages()
