import tarfile

import sandbox.common.types.resource as ctr
from sandbox import sdk2


class BigmodRunCommandWithResource(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Resource') as resource_group:
            resource_by_id = sdk2.parameters.Bool('Resource by id', default=False)
            unpack = sdk2.parameters.Bool('Unpack resource', default=False)
            with resource_by_id.value[False]:
                resource_type = sdk2.parameters.String('Resource type', required=True)
                resource_owner = sdk2.parameters.String('Resource owner', default=None)
                resource_attrs = sdk2.parameters.Dict('Resource attrs')
            with resource_by_id.value[True]:
                resource = sdk2.parameters.Resource('Resource', required=True)

        with sdk2.parameters.Group('Command') as command_group:
            binary = sdk2.parameters.String(
                'Binary',
                description='Executable file template using {resource}',
                required=True,
                default_value='{resource}',
            )
            args = sdk2.parameters.Dict('args', description='Command arguments')
            environment = sdk2.parameters.Dict(
                'Environment',
                description='Environment variables (support yav secrets)',
            )

    def on_execute(self):
        environment = self._parse_environment_variables()

        resource_id, resource_path = self._get_resource()

        self.set_info(
            'Used <a href="https://sandbox.yandex-team.ru/resource/{}">resource</a>'.format(resource_id),
            do_escape=False,
        )

        command = self._create_command(resource_path)

        with sdk2.helpers.ProcessRegistry, sdk2.helpers.ProcessLog(self, logger='main') as pl:
            sdk2.helpers.subprocess.check_call(
                command,
                env=environment,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

    def _parse_environment_variables(self):
        env_variables = {}

        for key, value in self.Parameters.environment.items():
            if value.startswith('sec-'):
                sec, token = value.split(':', 1)
                value = sdk2.yav.Secret(sec).data()[token]
            elif value.startswith('vault:'):
                _, owner, name = value.split(':', 2)
                value = sdk2.Vault.data(owner, name)
            env_variables[key] = value

        return env_variables

    def _get_resource(self):
        if self.Parameters.resource_by_id:
            resource = self.Parameters.resource
        else:
            resource = sdk2.Resource.find(
                type=self.Parameters.resource_type,
                owner=self.Parameters.resource_owner,
                state=ctr.State.READY,
                attrs=self.Parameters.resource_attrs,
            ).order(-sdk2.Resource.id).first()

        if self.Parameters.unpack:
            resource_path = self.path().as_posix()
            tar_path = sdk2.ResourceData(resource).path.as_posix()
            with tarfile.open(tar_path) as tar:
                tar.extractall(path=resource_path)
        else:
            resource_path = sdk2.ResourceData(resource).path.as_posix()

        return resource.id, resource_path

    def _create_command(self, resource_path):
        command = [self.Parameters.binary.format(resource=resource_path)]
        for kv in self.Parameters.args.iteritems():
            command.extend(kv)
        return list(map(str, command))
