# coding: utf-8

import os
import sandbox.projects.bigmod.resources as bigmod_resources

from sandbox import sdk2
from sandbox.sandboxsdk import parameters, paths, process
from sandbox.projects.bigmod.common.base_bin_build import BaseBuildBigmodBins, init


class BuildBigmodPackage(BaseBuildBigmodBins):
    '''
    Binaries, configs and scripts for bigmod.
    '''

    type = 'BUILD_BIGMOD_PACKAGE'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 70 * 1024

    class BuildFuryConfigs(parameters.SandboxBoolParameter):
        name = 'build_fury_configs'
        description = 'fury configs'
        default_value = True
        group = 'Bigmod Files'

    class IncludeSvnConfigs(parameters.SandboxBoolParameter):
        name = 'include_svn_configs'
        description = 'configs'
        default_value = True
        group = 'Bigmod Files'

    class IncludeSvnScripts(parameters.SandboxBoolParameter):
        name = 'include_svn_scripts'
        description = 'scripts'
        default_value = True
        group = 'Bigmod Files'

    class IncludeSvnLoupeFrontend(parameters.SandboxBoolParameter):
        name = 'include_svn_loupe_frontend'
        description = 'loupe frontend'
        default_value = True
        group = 'Bigmod loupe frontend files'

    class BuildModelsCommon(parameters.SandboxBoolParameter):
        name = 'build_models_common'
        description = 'common models'
        default_value = True
        group = 'Bigmod models'

    class BuildModelsMultik(parameters.SandboxBoolParameter):
        name = 'build_models_multik'
        description = 'multik models'
        default_value = True
        group = 'Bigmod models'

    class BuildBigmodUdfBase(parameters.SandboxBoolParameter):
        target_resource = bigmod_resources.BIGMOD_UDF_BASE
        name = 'build_bigmod_udf'
        description = 'bigmod_udf'
        default_value = False
        group = 'Bigmod UDFs'

    TARGET_RESOURCES = (
        bigmod_resources.BIGMOD_MODHUB,
        bigmod_resources.BIGMOD_GATEKEEPER,
        bigmod_resources.BIGMOD_FUNNEL,
        bigmod_resources.BIGMOD_HARVESTER,
        bigmod_resources.BIGMOD_LOUPE,
        bigmod_resources.BIGMOD_DELAYED_AGENT,
        bigmod_resources.BIGMOD_DISPERSER,
        bigmod_resources.BIGMOD_TAXONOMY,
        bigmod_resources.BIGMOD_INDEXER,
        bigmod_resources.BIGMOD_COMPRESSOR,
        bigmod_resources.BIGMOD_REVISER,
        # Fury
        bigmod_resources.BIGMOD_FURY,
        # Rays
        bigmod_resources.BIGMOD_LBEXPORT,
        bigmod_resources.BIGMOD_TRANSCEIVER,
        bigmod_resources.BIGMOD_HTTPEXCHANGE,
        bigmod_resources.BIGMOD_SERAPHIM,
        bigmod_resources.BIGMOD_DUMMY_FACTORY,
        # Tools
        bigmod_resources.BIGMOD_CREATE_INSTANCE_TOOL,
        bigmod_resources.BIGMOD_STATIC_EVENTS_WRITER_TOOL,
        bigmod_resources.BIGMOD_SATELLITE_TOOL,
        bigmod_resources.BIGMOD_REALTIME_MONITORING_TOOL,
    )

    FURY_BUILD_OUTPUTS = (
        'quality/antifraud/xurma/configs/bigmod/uxl/combos.json',
        'quality/antifraud/xurma/configs/bigmod/uxl/features.json',
        'quality/antifraud/xurma/configs/bigmod/uxl/rules.json',
    )

    FURY_SVN_CONFIGS = (
        'quality/antifraud/xurma/configs/bigmod/graph.json',
    )

    input_parameters, enabled_builds = init(
        TARGET_RESOURCES,
        [
            BuildFuryConfigs,
            IncludeSvnConfigs,
            IncludeSvnScripts,
            IncludeSvnLoupeFrontend,
            BuildModelsCommon,
            BuildModelsMultik,
            BuildBigmodUdfBase,
        ],
    )

    def _create_fury_configs_resource(self, source_dir, output_dir):
        fury_configs_dir_name = 'fury'
        fury_configs_dir = paths.make_folder(fury_configs_dir_name)

        for fury_config_item in self.FURY_BUILD_OUTPUTS:
            item_path = os.path.join(output_dir, fury_config_item)
            paths.copy_path(item_path, os.path.join(fury_configs_dir, os.path.basename(fury_config_item)))

        for fury_config_item in self.FURY_SVN_CONFIGS:
            item_path = os.path.join(source_dir, fury_config_item)
            paths.copy_path(item_path, os.path.join(fury_configs_dir, os.path.basename(fury_config_item)))

        paths.copy_path(os.path.join(output_dir, 'modadvert/bigmod/conf/patched'), os.path.join(fury_configs_dir, 'patched'))

        fury_configs_archive = os.path.abspath('fury_configs.tar.gz')
        process.run_process(
            [
                'tar',
                '-C', os.path.dirname(fury_configs_dir),
                '-zcf', fury_configs_archive,
                os.path.basename(fury_configs_dir),
            ],
            log_prefix='tar_fury_configs'
        )
        resource = self.create_resource(
            description='Build fury configs',
            resource_path=fury_configs_archive,
            resource_type=bigmod_resources.BIGMOD_FURY_CONFIGS,
        )
        self.mark_resource_ready(resource.id)

    def get_targets(self):
        targets = BaseBuildBigmodBins.get_targets(self)
        if self.ctx.get('build_fury_configs', False):
            targets.extend([
                'modadvert/bigmod/conf/patched',
                'quality/antifraud/xurma/configs/bigmod'
            ])
        if self.ctx.get('build_models_common', False):
            targets.extend([
                'modadvert/bigmod/models/common',
            ])
        if self.ctx.get('build_models_multik', False):
            targets.extend([
                'modadvert/bigmod/models/multik',
            ])
        return targets

    def on_execute(self):
        BaseBuildBigmodBins.on_execute(self)

    def post_build(self, source_dir, output_dir, pack_dir):
        BaseBuildBigmodBins.post_build(self, source_dir, output_dir, pack_dir)
        if self.ctx['include_svn_configs']:
            self._create_tar_resource(
                source_dir,
                [
                    ('modadvert/bigmod/conf', 'conf'),
                    ('modadvert/programs/dyn_disclaimers/config/dyn_disclaimers.yaml', 'conf/conf-common/dyn_disclaimers.yaml')
                ],
                'svn_configs.tar.gz',
                bigmod_resources.BIGMOD_SVN_CONFIGS,
            )
        if self.ctx['include_svn_scripts']:
            self._create_tar_resource(source_dir, [('modadvert/bigmod/scripts', 'scripts')], 'scripts.tar.gz', bigmod_resources.BIGMOD_SVN_SCRIPTS)
        if self.ctx['include_svn_loupe_frontend']:
            self._create_tar_resource(source_dir, [('modadvert/bigmod/loupe/frontend', 'frontend')], 'loupe_frontend.tar.gz', bigmod_resources.BIGMOD_SVN_LOUPE_FRONTEND)
        if self.ctx['build_models_common']:
            self._create_tar_resource(output_dir, [('modadvert/bigmod/models/common', 'models/common')], 'models_common.tar.gz', bigmod_resources.BIGMOD_MODELS_COMMON)
        if self.ctx['build_models_multik']:
            self._create_tar_resource(output_dir, [('modadvert/bigmod/models/multik', 'models/multik')], 'models_multik.tar.gz', bigmod_resources.BIGMOD_MODELS_MULTIK)
        if self.ctx['build_fury_configs']:
            self._create_fury_configs_resource(source_dir, output_dir)


__Task__ = BuildBigmodPackage
