from collections import defaultdict
from itertools import groupby, chain
from operator import itemgetter
from six.moves import map

MOODS = ['EAM_FORCE_GOOD', 'EAM_GOOD', 'EAM_SUSPICIOUS', 'EAM_BAD', None]
ITEM_TEMPLATE = '{prestable} !!(green)+{add}!! !!(red)-{remove}!! (!!({color}){diff_perc:+.2%}!!)'
TOTAL_ITEM_TEMPLATE = '!!(gray){diff}!! / {count} ({diff_perc:.2%})'


def get_moods_stat(rows):
    moods_stat = defaultdict(lambda: {'prestable': 0, 'production': 0, 'add': 0, 'remove': 0})
    total_count = 0
    total_diff = 0

    for row in rows:
        count = row['count']
        moods_stat[row['prestable_mood']]['prestable'] += count
        moods_stat[row['production_mood']]['production'] += count
        total_count += count
        if row['prestable_mood'] != row['production_mood']:
            moods_stat[row['prestable_mood']]['add'] += count
            moods_stat[row['production_mood']]['remove'] += count
            total_diff += count

    for mood, stat in moods_stat.items():
        diff = float(stat['add'] - stat['remove'])
        if stat['production']:
            stat['diff_perc'] = diff / stat['production']
        else:
            stat['diff_perc'] = float('inf') * diff if diff else 0.

    moods_stat['total'] = {
        'count': total_count,
        'diff': total_diff,
        'diff_perc': float(total_diff) / total_count
    }
    return moods_stat


def get_flags_stat(rows):
    flags_stat = defaultdict(lambda: {'prestable': 0, 'production': 0, 'add': 0, 'remove': 0})
    total_count = 0
    total_diff = 0

    for row in rows:
        count = row['count']
        flags_stat[row['prestable_is_set']]['prestable'] += count
        flags_stat[row['production_is_set']]['production'] += count
        total_count += count
        if row['prestable_is_set'] != row['production_is_set']:
            flags_stat[row['prestable_is_set']]['add'] += count
            flags_stat[row['production_is_set']]['remove'] += count
            total_diff += count

    for is_set, stat in flags_stat.items():
        diff = float(stat['add'] - stat['remove'])
        if stat['production']:
            stat['diff_perc'] = diff / stat['production']
        else:
            stat['diff_perc'] = float('inf') * diff if diff else 0.

    flags_stat['total'] = {
        'count': total_count,
        'diff': total_diff,
        'diff_perc': float(total_diff) / total_count
    }
    return flags_stat


def get_message_item(stat, tolerance=0.05):
    if not stat:
        return ''
    diff_perc = stat['diff_perc']
    return ITEM_TEMPLATE.format(
        color='gray' if abs(diff_perc) < tolerance else 'green' if diff_perc > 0. else 'red',
        **stat
    )


def get_stat_table(stat_type, key_fields, sorted_rows, highlight_tolerance=0.05, hide_tolerance=-1):
    assert hide_tolerance < highlight_tolerance
    assert stat_type in ('mood', 'flag')

    stat_list = []
    for keys, rows in groupby(sorted_rows, itemgetter(*key_fields)):
        if stat_type == 'mood':
            stat_list.append((list(keys) if isinstance(keys, tuple) else [keys], get_moods_stat(rows)))
        else:
            stat_list.append((list(keys) if isinstance(keys, tuple) else [keys], get_flags_stat(rows)))

    message_lines = []
    def add_message_line(*line_items):
        message_lines.append('|| {} ||'.format(' | '.join(map(str, chain.from_iterable(line_items)))))

    message_lines.append('#|')

    if stat_type == 'mood':
        add_message_line(key_fields, MOODS, ['total'])
    else:
        add_message_line(key_fields, ['count'])

    for keys, stat in stat_list:
        if stat['total']['diff_perc'] > hide_tolerance:
            if stat_type == 'mood':
                add_message_line(
                    keys,
                    (get_message_item(stat.get(mood), highlight_tolerance) for mood in MOODS),
                    [TOTAL_ITEM_TEMPLATE.format(**stat['total'])]
                )
            else:
                add_message_line(
                    keys,
                    [get_message_item(stat.get(True), highlight_tolerance)]
                )
            if stat['total']['diff_perc'] > highlight_tolerance:
                message_lines[-1] = '**{}**'.format(message_lines[-1])
    message_lines.append('|#')

    return '\n'.join(message_lines)
