# coding: utf-8

import logging
import os
import stat
import tarfile

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToYaDeployTask
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToYaDeployParameter
import sandbox.projects.common.build.parameters as build_params


def target2field_name(target):
    return 'build_%s' % target


class BigmodBinaries(parameters.SandboxInfoParameter):
    name = 'info_field'
    description = 'Bigmod Binaries'


class CreateSymbolsArchive(parameters.SandboxBoolParameter):
    name = 'bigmod_create_symbols_archive'
    description = 'Create separate tarball with debug symbols'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class StripBinaries(parameters.SandboxBoolParameter):
    name = 'bigmod_strip_binaries'
    description = 'Strip result binaries'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuse(build_params.UseArcadiaApiFuse):
    default_value = True


class YtStore(build_params.YtStore):
    default_value = True


def init(target_resources, custom_input_parameters):
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        CreateSymbolsArchive,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
        build_params.DefinitionFlags,
        ReleaseToYaDeployParameter,
    ]

    enabled_builds = {}
    input_parameters.append(BigmodBinaries)
    for target_resource in target_resources:
        target = os.path.basename(target_resource.arcadia_build_path)
        field_name = target2field_name(target)
        build = type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )
        input_parameters.append(build)
        enabled_builds[field_name] = target_resource

    for param in custom_input_parameters:
        if hasattr(param, "target_resource"):
            enabled_builds[param.name] = param.target_resource

    if custom_input_parameters:
        input_parameters.extend(custom_input_parameters)
    return input_parameters, enabled_builds


class BaseBuildBigmodBins(YaMakeTask, ReleaseToYaDeployTask):
    type = 'BASE_BUILD_BIGMOD_BINS'

    enabled_builds = {}
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
    ]
    TARGET_RESOURCES = ()

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM
        self.ctx[StripBinaries.name] = False
        # self.ctx[CreateSymbolsArchive.name] = True

    def get_targets(self):
        targets = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                targets.append(os.path.dirname(resource_type.arcadia_build_path))
        return targets

    def get_resources(self):
        resources = {}
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                resource = {
                    'description': resource_type.name,
                    'resource_type': resource_type,
                    'resource_path': os.path.basename(resource_type.arcadia_build_path),
                }
                resources[resource_type.name] = resource
        return resources

    def get_arts(self):
        arts = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                arts.append({'path': resource_type.arcadia_build_path})
        return arts

    def __copy_symbols(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--only-keep-debug', object_file, symbols_file],
            log_prefix='copy_symbols'
        )

    def __strip_binary(cls, object_file):
        mode = os.stat(object_file).st_mode
        os.chmod(object_file, mode | stat.S_IWUSR)
        process.run_process(
            ['strip', '-g', object_file],
            log_prefix='strip_binary'
        )
        os.chmod(object_file, mode)

    def __link_symbols_with_executable(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--add-gnu-debuglink', symbols_file, object_file],
            log_prefix='debuglink'
        )

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

        self.__process_symbols(pack_dir)

    def get_yp_oauth_token(self):
        owner, vault_name = 'BIGMOD_TEAM', 'robot-bigmod-yp-token'
        try:
            return self.get_vault_data(owner, vault_name)
        except Exception:
            raise RuntimeError("Cannot load yp token from vault. Owner: {}, Vault: {}".format(owner, vault_name))

    def __process_symbols(self, pack_dir):
        create_symbols_archive = self.ctx[CreateSymbolsArchive.name]
        strip_symbols = self.ctx[StripBinaries.name]

        symbols_dir_name = 'symbols'
        symbols_dir = paths.make_folder(symbols_dir_name)

        for build_name, resource_type in self.enabled_builds.items():
            if not self.ctx.get(build_name):
                continue

            binary_name = os.path.basename(resource_type.arcadia_build_path)
            binary_path = os.path.join(pack_dir, binary_name)
            symbols_file = os.path.join(symbols_dir, binary_name + '.sym')

            if create_symbols_archive:
                self.__copy_symbols(binary_path, symbols_file)

            if strip_symbols or getattr(resource_type, 'always_strip', False):
                self.__strip_binary(binary_path)

            if create_symbols_archive:
                self.__link_symbols_with_executable(binary_path, symbols_file)

        if not create_symbols_archive:
            return

        symbols_archive_file = os.path.abspath('symbols.tar.gz')

        process.run_process(
            ['tar', 'czf', symbols_archive_file, symbols_dir_name],
            log_prefix='tar_symbols'
        )

        resource = self.create_resource(
            description='Build symbols',
            resource_path=symbols_archive_file,
            resource_type=resource_types.BIGMOD_SYMBOLS_ARCHIVE,
            attributes={
                'ttl': 4
            }
        )

        self.mark_resource_ready(resource.id)

    def on_execute(self):
        self.ctx['ya_yt_token_vault_owner'] = 'BIGMOD_TEAM'
        self.ctx['ya_yt_token_vault_name'] = 'robot-bigmod-yt-token'
        YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        YaMakeTask.on_release(self, additional_parameters)
        ReleaseToYaDeployTask.on_release(self, additional_parameters)

    def _create_tar_resource(self, data_dir, data_destinations, resource_name, resource_type):
        logging.info('Creating tar resource: %s (%s)', resource_name, resource_type)
        with tarfile.open(resource_name, "w:gz") as tar:
            for source, destination in data_destinations:
                logging.info('  %s => %s', source, destination)
                tar.add(os.path.join(data_dir, source), arcname=destination)
        self.create_resource('', resource_name, resource_type)
