# -*- coding: utf-8 -*-

import logging

from sandbox.projects.common import binary_task
from sandbox.projects.resource_types import OTHER_RESOURCE
from sandbox.projects.billing import resources
from sandbox.projects.billing.common import enqueue_fuse_task, get_module_from_path
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox import sdk2


class BillingFaasBuildPythonTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Проверяет существует ли `FAAS_RESOURCE` с введенными данным и создает новый ресурс(OTHER_RESOURCE) c ttl=1,
     если в SB такого ресурса ещё нет. Ресурс предназначен для копирования таской BILLING_FAAS_BUILD_TASK.
    """

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = "billing_faas_build_python_task"

        with sdk2.parameters.Group("Build parameters") as build_params:
            peerdirs = sdk2.parameters.String("Peerdirs", required=True)
            functions = sdk2.parameters.String("Functions", required=True)
            revision = sdk2.parameters.Integer("Revision", required=True)
            instance = sdk2.parameters.String("Instance", required=True)
            namespace = sdk2.parameters.String("Namespace", required=False)
            tenant = sdk2.parameters.String("Tenant", required=True)
            pr_id = sdk2.parameters.Integer("PR ID")

            change_binary = sdk2.parameters.Bool("[Debug] Change default binary for faas", default=False)
            with change_binary.value["True"]:
                binary = sdk2.parameters.String(
                    "Path to the binary",
                    default="billing/hot/faas/python/bin/faas",
                    required=True,
                )

        with sdk2.parameters.Group("Task executor") as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output:
            output_resource = sdk2.parameters.Resource("Faas resource", required=True)

    class Context(sdk2.Task.Context):
        build_task_id = None
        copy_task_id = None

    def on_execute(self):
        peerdirs = self.Parameters.peerdirs
        functions = self.Parameters.functions
        revision = self.Parameters.revision
        instance = self.Parameters.instance
        pr_id = self.Parameters.pr_id
        tenant = self.Parameters.tenant

        binary_module = get_module_from_path(self.Parameters.binary)

        def title():
            return "{tenant}:{instance}#{revision}".format(tenant=tenant, instance=instance, revision=revision)

        with self.memoize_stage.first_stage:
            self.Parameters.description = "Checking for saved resources"

            if not all((peerdirs, functions, revision, instance, tenant)):
                raise Exception("peerdirs, functions, instance, tenant and revision are required")

            attrs = {
                "revision": revision,
                "peerdirs": peerdirs,
            }
            if pr_id:
                attrs["patch"] = pr_id

            saved_res = resources.FaasResource.find(attrs=attrs).first()

            if saved_res:
                saved_res_pr_id = int(getattr(saved_res, "patch", None) or 0)

                if saved_res_pr_id == pr_id:
                    logging.info("using cached resource: {}".format(saved_res.id))
                    self.Parameters.description = "Used saved resource for {title}".format(title=title())
                    self._save_resource(saved_res)
                    return
            else:
                logging.info("no saved resources found, starting build")

            self.Parameters.description = "Building FaaS for {title}".format(title=title())
            build_kwargs = {
                "description": "Building {title} with peerdirs {peerdirs}".format(peerdirs=peerdirs, title=title()),
                "owner": self.owner,
                "checkout_arcadia_from_url": "arcadia:/arc/trunk/arcadia",
                "program": self.Parameters.binary,
                "program_args": "check",
                "result_single_file": True,
                "resource_dir": binary_module,
                "env_vars": "YENV_TYPE=development YT_TOKEN='$(vault:value:BILLING-CI:yt-token)' "
                "FUNCTION={functions}".format(functions="'{}'".format(functions)),
                "definition_flags": "-DFUNCTION_PEERDIR={peerdirs}".format(peerdirs=peerdirs),
            }

            if revision:
                build_kwargs["checkout_arcadia_from_url"] = "{checkout_arcadia_from_url}@{revision}".format(
                    checkout_arcadia_from_url=build_kwargs["checkout_arcadia_from_url"],
                    revision=revision,
                )

            if pr_id:
                build_kwargs["arcanum_review_id"] = "arc:{pr_id}".format(pr_id=pr_id)

            build_task = enqueue_fuse_task(self, build_kwargs)
            self.Context.build_task_id = build_task.id
            raise sdk2.WaitTask(build_task, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])

        with self.memoize_stage.second_stage:
            self.Parameters.description = "Built FaaS for {title}".format(title=title())

            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in ctt.Status.Group.SUCCEED:
                raise Exception("Instance `{}` is not valid".format(instance))

            build_resource = OTHER_RESOURCE.find(
                task=sdk2.Task[self.Context.build_task_id], state=ctr.State.READY
            ).first()

            if build_resource:
                build_resource.ttl = 1  # we gonna copy
                self._save_resource(build_resource)
            else:
                raise Exception("Build resource was not created")

    def _save_resource(self, resource):
        self.Parameters.output_resource = resource
