# -*- coding: utf-8 -*-
import json
import logging
import os
import shutil
import yaml

from sandbox.projects.billing.tasks.Faas.BillingFaasBuildTask.report import load_template, report_is_not_ready
from sandbox.projects.billing.tasks.Faas.BillingFaasBuildTask.core import ManifestValidatorCore

from sandbox.projects.billing.tasks.Faas.BillingFaasBuildTask.schemas import build_task_config_schema

from sandbox.projects.resource_types import OTHER_RESOURCE, TASK_CUSTOM_LOGS
from sandbox.projects.common import binary_task
from sandbox import sdk2

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects.billing import resources
from sandbox.projects.billing.tasks.Manificenta.BillingManifestsGenerationTask import BillingManifestsGenerationTask

FAAS_SECTION_NAME = "faas"
MANIFESTS_FILE_NAME = "manifests.yml"
TASK_URL = "https://sandbox.yandex-team.ru/task/{id}/"
MINIMAL_REVISION = 8689402

# Convertion from manificenta format to DeployTasklet format.
ENVIRONMENTS = {
    "dev": "development",
    "test": "testing",
    "prod": "production",
}


class NamespaceAdaptor(logging.LoggerAdapter):
    """
    NamespaceAdaptor adds context with manifest namespace to logger.
    """

    def process(self, msg, kwargs):
        return "[%s] %s" % (self.extra["namespace"], msg), kwargs


class BillingFaasBuildTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Таска, занимающаяся сборкой бинарных файлов для faas.
    Запускает манифисенту для нужного сервиса (см. https://a.yandex-team.ru/arc/trunk/arcadia/billing/hot/manificenta)
     далее проходит по полученным манифестам и агрегирует калькуляторы по инстансам и тенантам,
     после чего запускает таску `BILLING_FAAS_BUILD_PYTHON_TASK` для сборки каждого инстанса,
     и в конце сохраняет полученные бинарные файлы в `FAAS_RESOURCE`.
    При возникновении ошибки на любом этапе - происходит остановка работы.
    Документация: https://docs.yandex-team.ru/billing/faas/platform#billingfaasbuildtask.
    """

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = "billing_faas_build_task"

        with sdk2.parameters.Group("Build parameters") as build_params:
            with sdk2.parameters.RadioGroup("Enviroment", group="Build parameters") as env:
                env.values["test"] = env.Value("test", default=True)
                env.values["dev"] = env.Value("development")
                env.values["prod"] = env.Value("production")

            with sdk2.parameters.RadioGroup("Service", group="Build parameters") as service:
                service.values["processor"] = service.Value("processor", default=True)
                service.values["mediator"] = service.Value("mediator")
                service.values["editor"] = service.Value("editor")
                service.values["faas"] = service.Value("faas")

            validate_revision = sdk2.parameters.Bool("Validate faas section revision", default=False)
            with validate_revision.value["True"]:
                minimal_revision = sdk2.parameters.Integer("Minimal revision", required=True, default=MINIMAL_REVISION)

            change_manificenta_setup = sdk2.parameters.Bool(
                "[Debug] Change default setup for manificenta", default=False
            )
            with change_manificenta_setup.value["True"]:
                manificenta_resource_id = sdk2.parameters.Integer("Manificenta resource id")
                manificenta_revision = sdk2.parameters.String(
                    "Manificenta revision",
                    default="-",
                    required=False,
                )

            dry_run = sdk2.parameters.Bool("Dry run", default=False)

            arcadia_hash = sdk2.parameters.String("Arcadia hash")

            override_manifests = sdk2.parameters.Bool(
                "Use custom manifests instead of generated by Manificenta",
                default=False,
            )

            with override_manifests.value["True"]:
                manifests = sdk2.parameters.String(
                    "Custom manifests",
                    default="",
                    multiline=True,
                    required=True,
                )

        with sdk2.parameters.Group("Task executor") as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        # Manificenta task id is used to get task resources after YA_EXEC.
        manificenta_task_id = None

        # Tasks configs are used to resave resource with parameters that were passed to tasks.
        # Schema: 'task_id': {'build_config': <build_task_config>, 'variables': <additional_variables>}
        tasks_configs = dict()

        # Cached HTML report for parsed tenants.
        # Report method is invoked every time user opens report tab, so we cache report requiring less resourses.
        tenant_report = report_is_not_ready

    # Logger is used to support adaptors.
    # During parsing the manifests we use logging.LoggerAdaptor instead of logging.Logger
    logger = None

    # Log file template.
    default_task_log_file = "{}.log"

    @sdk2.report(title="Faas report", label="report")
    def report(self):
        """
        Return report on built calculators.
        This method is invoked every time user opens 'Faas report' tab,
         so we render report in the task itself and cache it to use less resources.
        By default, we return 'Report is not ready' message.
        :return: HTML string serialized report.
        """
        return self.Context.tenant_report

    def prepare_task_logger(self, stage):
        """
        Prepare logger. It is done every time task is reloaded.
        Logger "task" is going to log to a file <stage>.log and this file will be published as TASK_CUSTOM_LOGS.
        This increases readability of task logs.

        :param stage: name of the stage.
        :return:
        """
        log_resource = TASK_CUSTOM_LOGS(self, "task logs", self.default_task_log_file.format(stage))
        log_file_path = str(sdk2.ResourceData(log_resource).path)

        logger = logging.getLogger("task")
        fh = logging.FileHandler(log_file_path)
        fh.setLevel(logging.DEBUG)
        fm = logging.Formatter("%(asctime)s %(levelname)-8s (%(name)s) %(message)s")
        fh.setFormatter(fm)
        logger.addHandler(fh)
        self.logger = logger
        self.logger.info("initiated logger for stage: %s", stage)

    def on_execute(self):
        service = self.Parameters.service
        env = self.Parameters.env

        if not all((service, env)):
            raise Exception("Service and env are required")

        if not self.Parameters.override_manifests:
            with self.memoize_stage.generate_manifests:
                self.Parameters.description = "Generating manifests"
                self.prepare_task_logger("generate_manifests")

                generate_kwargs = {"env": env, "service": service}

                if self.Parameters.change_manificenta_setup:
                    if self.Parameters.manificenta_resource_id:
                        generate_kwargs["resource_id"] = self.Parameters.manificenta_resource_id
                    if self.Parameters.manificenta_revision != "-":
                        generate_kwargs["revision"] = self.Parameters.manificenta_revision
                        self.logger.warning(
                            "not using arc HEAD for Manificenta, revision to use: %s",
                            self.Parameters.manificenta_revision,
                        )

                if self.Parameters.arcadia_hash:
                    generate_kwargs["arcadia_hash"] = self.Parameters.arcadia_hash
                    self.logger.warning(
                        "not using arc HEAD for Manificenta, arcadia hash: %s",
                        self.Parameters.arcadia_hash,
                    )

                self.logger.debug(f"generating manificents with parameters: {generate_kwargs}")

                generate_task = BillingManifestsGenerationTask(self, **generate_kwargs).save().enqueue()
                self.Context.manificenta_task_id = generate_task.id
                raise sdk2.WaitTask(generate_task, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])

        with self.memoize_stage.parse_manifests:
            self.Parameters.description = "Parsing manifests"
            self.prepare_task_logger("parse_manifests")

            if self.Parameters.override_manifests:
                self.logger.debug("using manifests from parameters")
                manifests = yaml.safe_load(self.Parameters.manifests)
            else:
                self.logger.debug("using manifests generated by manificenta")
                generate_task = sdk2.Task[self.Context.manificenta_task_id]
                if generate_task.status not in ctt.Status.Group.SUCCEED:
                    raise Exception(
                        "Could not finish building and running manificenta task: {id}".format(id=generate_task.id)
                    )
                generate_resource = sdk2.resource.Resource.find(
                    task=generate_task, state=ctr.State.READY, resource_type=OTHER_RESOURCE
                ).first()

                generate_resource_data = sdk2.ResourceData(generate_resource)
                with (generate_resource_data.path.joinpath(MANIFESTS_FILE_NAME)).open() as yaml_stream:
                    manifests = yaml.safe_load(yaml_stream)

            self.handle_tenants(manifests)

            if not self.Parameters.dry_run:
                raise sdk2.WaitTask(
                    list(self.Context.tasks_configs.keys()),
                    [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                    wait_all=True,
                )

        with self.memoize_stage.save_faas_resources:
            self.Parameters.description = "Saving faas resources"
            self.prepare_task_logger("save_resources")

            if self.Parameters.dry_run:
                self.logger.info("[dry run] skipping handling build tasks")
                return

            for task_id, task_config in self.Context.tasks_configs.items():
                build_task = sdk2.Task[task_id]
                if build_task.status in ctt.Status.Group.SUCCEED:
                    self._handle_build_task_output(build_task, task_config)
                else:
                    self.Parameters.description = "Exception occurred in subtasks"
                    raise Exception(
                        "Task {task} was not succeeded, details: {task_url}".format(
                            task=build_task, task_url=TASK_URL.format(id=task_id)
                        )
                    )
            self.Parameters.description = "Build completed"

    def parse_and_validate_tenants(self, manificents_config):
        manifest_core = ManifestValidatorCore(self.Parameters.validate_revision, self.Parameters.minimal_revision)
        return manifest_core.parse_and_validate_tenants(manificents_config)

    def handle_tenants(self, manificents_config):
        FaasBuildPythonTask = sdk2.Task["BILLING_FAAS_BUILD_PYTHON_TASK"]  # noqa

        tenants = self.parse_and_validate_tenants(manificents_config)

        for tenant_name, tenant_settings in tenants.items():
            for instance_name, instance_data in tenant_settings["instances"].items():
                instance_endpoints = instance_data["endpoints"]
                instance_settings = instance_data["settings"]

                functions = [
                    {
                        "name": endpoint["endpoint"],
                        "function": endpoint["function"],
                        "settings": endpoint["settings"],
                    }
                    for endpoint in instance_endpoints
                ]

                build_task_config = {
                    "tenant": tenant_name,
                    "functions": build_task_config_schema.dumps(functions, many=True),
                    "peerdirs": '"{endpoints}"'.format(
                        endpoints=" ".join(
                            sorted(set([endpoint["peerdir"] for endpoint in instance_endpoints]))
                            # Set is sorted to avoid difference in binary hashes with differently ordered peerdirs.
                            # See: https://st.yandex-team.ru/BILLING-1285
                        )
                    ),
                    "revision": instance_endpoints[0]["revision"],
                    "instance": instance_name,
                    "instance_settings": json.dumps(instance_settings),
                }

                self.logger.debug(
                    f"tenant: {tenant_name};\n"
                    f"instance: {instance_name};\n"
                    f"build task parameters: {build_task_config}"
                )

                if self.Parameters.dry_run:
                    self.logger.info("[dry run] skip enqueuing task")
                    continue

                build_task = FaasBuildPythonTask(self, **build_task_config).save().enqueue()
                self.Context.tasks_configs[build_task.id] = {
                    "build_config": build_task_config,
                    "variables": {
                        "functions": json.dumps(functions),
                        "namespaces": json.dumps(tenant_settings["namespaces"]),
                    },
                }
                self.logger.debug(
                    "enqueued task for tenant `%s` instance `%s`",
                    tenant_name,
                    instance_name,
                )

        # save tenant report.
        self.Context.tenant_report = load_template(tenants)

    def _handle_build_task_output(self, task, faas_config):
        """
        Download faas files from children tasks and save them as own.
        This is the only way to deliver resourse from child task to parent task, that we found.
        We need to publish resourses in this task Output, and it's possible only when uploading resources from it.
        So, we have to recreate resoures even if using already created ones.

        :param task: task id.
        :param faas_config: config for the task that created faas resource.
        :return:
        """
        faas_filepaths = {resources.FaasResource: "faas", OTHER_RESOURCE: "bin/faas"}
        resource = task.Parameters.output_resource

        self.logger = NamespaceAdaptor(
            logger=logging.getLogger("task"),
            extra={"namespace": faas_config["build_config"]["tenant"]},
        )
        try:
            self._save_faas_resource(resource, faas_config, faas_filepaths[resource.type])
        except KeyError:
            self.logger.exception("resource type %s is not supported", resource.type)

        self.logger = logging.getLogger("task")

    def _save_faas_resource(self, resource, faas_config, faas_filepath):
        """
        Download faas from child task and save in folder ./<task_id>/faas

        :param resource: resource id.
        :param faas_config: faas task config.
        :param faas_filepath: path of the resource in the task. It can vary from task output.
         If task created resource with YA_EXEC_FUSE - it will be OTHER_RESOURCE saved as bin/faas,
         if task returned existing resource it will return FAAS_RESOURCE and binary file is saved as /faas.
        :return: None
        """

        # Download resource data from skynet to root folder.
        # We cannot use common_share.skynet_get as it does not work with Python2
        #  (see: https://st.yandex-team.ru/SKYDEV-2096)
        # common_share.skynet_get(resource.skynet_id, '.', logger=self.logger)
        with sdk2.helpers.ProcessLog(self, logger="task") as pl:
            sdk2.helpers.subprocess.check_call(
                ["sky", "get", "-wu", resource.skynet_id],
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

        # Create folder for faas binary and copy binary to it.
        # For now, if we need to save already existing resource,
        #  we save it to directory `./<resource_id>_<copy_number>/`
        # There are proposals for optimizing this behaviour. See: https://st.yandex-team.ru/BILLING-1334
        copy_counter = 0
        while True:
            task_folder = f"./{resource.id}" if copy_counter == 0 else f"./{resource.id}_{copy_counter}"
            try:
                os.mkdir(task_folder)
                break
            except FileExistsError:
                self.logger.warning("task folder `%s` already exists, using it", task_folder)
                copy_counter += 1

        shutil.move(f"./{faas_filepath}", task_folder)
        self.logger.debug("copied resource to folder %s", task_folder)

        build_config = faas_config["build_config"]
        variables = faas_config["variables"]

        # Create new FAAS_RESOURCE
        faas_resource = resources.FaasResource(
            task=self,
            description=f'FaaS for {build_config["tenant"]}-{build_config["instance"]}#r{build_config["revision"]}',
            ttl="inf",
            revision=build_config["revision"],
            peerdirs=build_config["peerdirs"],
            functions=variables["functions"],
            namespaces=variables["namespaces"],
            tenant=build_config["tenant"],
            instance=build_config["instance"],
            instance_settings=build_config["instance_settings"],
            current_environment=ENVIRONMENTS[self.Parameters.env],
            path=f"{task_folder}/faas",
        )

        self.logger.debug("created resource: %s", faas_resource.id)

        # Save the resource
        sdk2.ResourceData(faas_resource).ready()
