import logging
from sandbox.projects.billing.tasks.Faas.BillingFaasBuildTask.schemas import (
    faas_schema,
    namespace_faas_schema,
    config_schema,
)

FAAS_SECTION_NAME = "faas"


class NamespaceAdaptor(logging.LoggerAdapter):
    """
    NamespaceAdaptor adds context with manifest namespace to logger.
    """

    def process(self, msg, kwargs):
        return "[%s] %s" % (self.extra["namespace"], msg), kwargs


class ManifestValidatorCore:
    """
    Class is used to parse tenants from manificenta.
    It is separated from the task due to unit testing.
    """

    logger = logging.getLogger("task")

    minimal_revision = 0
    validate_revision = False

    def __init__(self, validate_revision, minimal_revision):
        self.validate_revision = validate_revision
        self.minimal_revision = minimal_revision

    def parse_and_validate_tenants(self, manificenta_config):
        """
        Handle manifests and separate endpoints by tenants and instances.

        :param manificenta_config: manificenta config with manifests.
        :return: None
        """

        manifests = manificenta_config["manifests"]
        global_faas_settings = manificenta_config["faas"]

        tenants_settings = self.validate_tenant_faas_settings(global_faas_settings)
        tenants = self.parse_tenants(manifests, tenants_settings)

        for tenant_name, instances in tenants.items():
            for instance_name, instance_data in instances["instances"].items():
                instance_endpoints = instance_data["endpoints"]
                self.validate_endpoints_in_instance(instance_name, instance_endpoints)

        return tenants

    def parse_tenants(self, manifests, tenants_settings):
        """
        Parse tenants from manifests.

        :param manifests: manifests from manificenta.
        :param tenants_settings: tenants settings from manificenta.
        :return: dictionary with valid tenants.
        """
        tenants = dict()

        for manifest in manifests:
            namespace = manifest["namespace"]

            self.logger = NamespaceAdaptor(logger=logging.getLogger("task"), extra={"namespace": namespace})
            logger = self.logger

            logger.debug("====== parsing manifest for namespace: `%s` ======", namespace)

            namespace_faas_config = self.validate_namespace_faas_configuration(manifest)
            logger.debug("namespace faas settings: %s", namespace_faas_config)

            namespace_tenants = namespace_faas_config["tenants"]
            logger.debug("namespace tenants are set to `%s`", namespace_tenants)
            for namespace_tenant in namespace_tenants:
                # namespace tenant settings can be updated later with instances that are not set explicitly.
                namespace_tenant_settings = tenants_settings.setdefault(namespace_tenant, {"instances": {}})
                logger.debug("namespace tenant settings: %s", namespace_tenant_settings)

                try:
                    instances = self.parse_instances(manifest["endpoints"], namespace, namespace_tenant)
                except Exception as exc:
                    raise Exception(
                        "Could not parse instances.\n"
                        "tenant: {tenant}\n"
                        "namespace: {namespace}\n"
                        "endpoints: {endpoints}\n"
                        "error: {error}".format(
                            tenant=namespace_tenant,
                            namespace=namespace,
                            endpoints=manifest.setdefault("endpoints", "not set"),
                            error=exc,
                        )
                    )

                for instance_name, instance_endpoints in instances.items():
                    logger.debug("parsing instance `%s`", instance_name)

                    if not instance_endpoints:
                        logger.warning(
                            "could not find endpoints for instance `%s`. Skipping instance.",
                            instance_name,
                        )
                        continue

                    instance_settings = self.validate_instance_settings(
                        namespace_tenant_settings["instances"], instance_name
                    )

                    logger.debug(
                        "instance settings for instance `%s`: %s",
                        instance_name,
                        instance_settings,
                    )

                    if not instance_settings["active"]:
                        logger.info(
                            "instance `%s` for tenant `%s` is not active",
                            instance_name,
                            namespace_tenant,
                        )
                        continue

                    self._update_tenant_instance(
                        tenants,
                        namespace_tenant,
                        namespace,
                        instance_name,
                        instance_settings,
                        instance_endpoints,
                    )

            logger.debug("====== finished parsing namespace `%s` ======", namespace)

        self.logger = logging.getLogger("task")

        for tenant_name, tenant_data in tenants.items():
            tenants[tenant_name]["namespaces"] = list(tenants[tenant_name]["namespaces"])

        return tenants

    def _update_tenant_instance(self, tenants, tenant, namespace, instance, instance_settings, instance_endpoints):
        """
        Update instance for given tenant.

        Schema:
          'tenants': {
            'tenant_name': {
              'instances': {
                'instance_name': {
                  'endpoints': [ {instance_endpoint_1}, {instance_endpoint_2} ],
                  'settings': { <instance_settings> }
                },
              },
              'namespaces': ['first_namespace', 'second_namespace'],
            }
          }

        :param tenants: tenants dict.
        :param tenant: tenant name
        :param instance: instance name
        :param instance_settings: instance settings.
        :param instance_endpoints: instance endpoints
        :return:
        """
        tenants.setdefault(tenant, {"instances": {}, "namespaces": set()})
        tenants[tenant]["namespaces"].add(namespace)
        tenants[tenant]["instances"].setdefault(instance, {"settings": instance_settings, "endpoints": []})
        tenants[tenant]["instances"][instance]["endpoints"].extend(instance_endpoints)

    def _get_tenant_settings(self, tenants_settings, tenant):
        """ """
        return tenants_settings.setdefault(tenant, {"instances": {}})

    def validate_endpoints_in_instance(self, instance, endpoints):
        """
        Validate endpoints in instance.
        See that they all have the same revision and if set, that they all have required revision.

        :param instance: instance name.
        :param endpoints: instance endpoints.
        :return:
        """
        revisions = set([endpoint["revision"] for endpoint in endpoints])

        if len(revisions) > 1:  # cannot be < 1, see: self._validate_endpoint_faas_section()
            raise Exception(
                "Instance `{instance}` contains more than one revision: {revisions}".format(
                    instance=instance, revisions=revisions
                )
            )

        if self.validate_revision:
            revision = int(list(revisions)[0])
            if revision < int(self.minimal_revision):
                raise Exception(
                    "Revision of instance `{instance}` is less than a minimal. {current} < {minimal}".format(
                        instance=instance,
                        current=revision,
                        minimal=self.minimal_revision,
                    )
                )

    def parse_instances(self, endpoints, namespace_name, tenant_name):
        """
        Parse instances from namespace.

        :param endpoints: endpoints from manificenta.
        :param: namespace_name: name of the namespace of endpoint
        :param tenant_name: name of the namespace tenant.
        :return: instances.
        """
        instances = dict()
        for endpoint_name, endpoint_data in endpoints.items():
            if FAAS_SECTION_NAME in endpoint_data:
                faas_section = endpoint_data[FAAS_SECTION_NAME]

                self.logger.info("endpoint `%s` has faas section: %s", endpoint_name, faas_section)

                endpoint_faas_config = self.validate_endpoint_faas_section(faas_section)

                # enrich config for BillingDeploy tasklet
                endpoint_faas_config["namespace"] = namespace_name
                endpoint_faas_config["tenant"] = tenant_name

                # https://st.yandex-team.ru/BILLING-1276
                # Add prefix if tenant is not default.
                endpoint_faas_config["endpoint"] = "-".join(
                    [namespace_name, endpoint_name] if namespace_name != tenant_name else [endpoint_name]
                )
                instances.setdefault(endpoint_faas_config["instance"], []).append(endpoint_faas_config)
            else:
                self.logger.info("endpoint `%s` does not have faas section", endpoint_name)
        return instances

    def validate_instance_settings(self, tenant_instance_settings, instance):
        """
        Validate instance settings. If tenant settings are not explicitly set, it means, that they are equal to default.

        Schema:
          see InstanceSchema.

        :param tenant_instance_settings: settings for instances.
        :param instance: current instance.
        :return: instance settings.
        """

        return tenant_instance_settings.setdefault(instance, {"active": True, "dcs": []})

    def validate_namespace_faas_configuration(self, manifest):
        """
        Validate faas configuration for namespace.
        Default tenant name equals to namespace, thus compatability with previous system is supported.

        Schema:
          'faas': {
            'tenant': 'tenant_name'
          }
        :param manifest: manifest to get faas config from.
        :return: faas configuration.
        """

        namespace = manifest["namespace"]
        faas_config = manifest.setdefault("faas", {})

        valid_faas_config = namespace_faas_schema.load(faas_config)

        if not valid_faas_config["tenants"]:
            valid_faas_config["tenants"].append(namespace)

        return valid_faas_config

    def validate_tenant_faas_settings(self, tenants_settings):
        """
        Validate tenant settings.
        Settings describe instances in the tenants and their DC distribution.

        Schema:
            {
              'tenant_name': {
                'instances': {
                  'instance_name': {
                    'active': True,
                    'dcs': [
                      {
                        'name': 'sas',
                        'amount': 1
                      }
                    ]
                  }
                }
              }
            }

        :param tenants_settings: tenants settings from manificenta.
        :return: validated settings.
        """

        valid_tenants_settings = config_schema.load(tenants_settings)

        self.logger.info("valid tenants settings: %s", valid_tenants_settings)

        tenants = valid_tenants_settings["tenants"]

        result = dict()
        for tenant in tenants:

            parsed_instances = dict()
            for instance in tenant["instances"]:
                parsed_instances[instance["name"]] = {
                    "active": instance["active"],
                    "dcs": instance["dcs"],
                }

            result[tenant["name"]] = {"instances": parsed_instances}

        self.logger.debug("parsed tenants settings: %s", result)

        return result

    def validate_endpoint_faas_section(self, faas_section):
        """
        Validate faas endpoint section.

        Schema:
          {
            'pr_id': 0,
            'function: 'billing.hot.faas.lib.dummy.dummy_calculator.dummy_calculator',
            'peerdir': 'billing/hot/faas/lib/dummy',
            'revision': 8689402,
            'instance': 'default',
            'settings': {
              'foo': 'bar'
            }
          }
        :param faas_section: faas section from manificenta.
        :return:
        """

        valid_faas_config = faas_schema.load(faas_section)
        return valid_faas_config
