# -*- coding: utf-8 -*-
import logging

from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.projects.common import binary_task
from sandbox import sdk2

from sandbox.projects.billing.tasks.Faas.BillingFaasBuildTask import BillingFaasBuildTask
from sandbox.projects.common.tasklet.executor import TaskletExecutor
from sandbox.projects.billing.resources.faas import FaasResource
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

# Constraint on the resources that task will pass from build task to tasklet.
FAAS_RESOURCE_LIMIT = 100


class BillingFaasDepotTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = "billing_faas_depot_task"

        deploy_id = sdk2.parameters.String("Deploy ID", required=True)

        with sdk2.parameters.Group("BILLING_FAAS_BUILD_TASK parameters") as billing_faas_build_task:
            build_input_parameters = sdk2.parameters.JSON(
                "Input parameters",
                default={
                    "env": "dev",
                    "service": "faas",
                    "validate_revision": True,
                    "minimal_revision": 8628496,
                },
                required=True,
            )

        with sdk2.parameters.Group("TASKLET_BILLING_FAAS_DEPLOY parameters") as tasklet_billing_faas_deploy:
            deploy_resource = sdk2.parameters.Resource("Tasklet resource", required=True)

            deploy_input_parameters = sdk2.parameters.JSON(
                "Input parameters",
                required=True,
                default={
                    "config": {
                        "release_type": "development",
                        "dry_run": True,
                        "stage_id": "billing-faas-dev-stage",
                        "awacs_namespace": "faas.test.billing.yandex.net",
                        "clusters": ["vla", "sas"],
                        "network_macro": "_BILLING_DEPLOY_FAAS_TEST_NETS_",
                        "logbroker_topics": {
                            "nginx_topic": "billing-tarifficator/faas/test/nginx",
                            "faas_topic": "billing-tarifficator/faas/test/faas",
                        },
                        "secrets": {
                            "tvm": {
                                "secret_uuid": "sec-01fctnbb3jamftzneev9dwt6sz",
                                "attributes": {"tvm_client_id": "2029827"},
                            },
                            "certificate": {"secret_uuid": "sec-01fctmfm1wxzt7chqzj22jn5x9"},
                        },
                    },
                    "context": {
                        # Yav secret that is used for tasklet to access YAV.
                        "secret_uid": "sec-01fd9rkb3xgkggbmsn5fjkw38m",
                    },
                },
            )

            # CI would use `secret` field from a.yaml and search for `ci.token` key in that secret
            #  that contains scopes [vault:use, sandbox:use] in there and then uses it
            #  in __tasklet_secret__ tasklet input parameter.
            # But TASKLET_RELEASER does not support YAV secrets,
            #  so we have to save our `ci.token` as Sandbox Vault entry and pass it as `yav_token_vault` parameter.
            # Tasklet uses `yav_token` only for VaultClient, but it has some other scopes.
            # USER(robot) that runs tasks must have access to Sandbox Vault entry via `shared`.
            tasklet_yav_token = sdk2.parameters.Vault("Tasklet YAV token", required=True)

        with sdk2.parameters.Group("Task executor") as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

        # Allows for using Python3 tasks.
        push_tasks_resource = True

    class Context(sdk2.Context):
        # child BILLING_FAAS_BUILD_TASK id.
        build_task_id = None

        # child TASKLET_EXECUTOR id.
        deploy_tasklet_id = None

    logger: logging.Logger

    def prepare_task_logger(self, stage):
        """
        Prepare logger. It is done every time task is reloaded.
        Logger "task" is going to log to a file <stage>.log and this file will be published as TASK_CUSTOM_LOGS.
        This increases readability of task logs.

        :param stage: name of the stage.
        :return:
        """
        log_resource = TASK_CUSTOM_LOGS(self, "task logs", f"{stage}.log")
        log_file_path = str(sdk2.ResourceData(log_resource).path)

        logger = logging.getLogger("task")
        fh = logging.FileHandler(log_file_path)
        fh.setLevel(logging.DEBUG)
        fm = logging.Formatter("%(asctime)s %(levelname)-8s (%(name)s) %(message)s")
        fh.setFormatter(fm)
        logger.addHandler(fh)
        self.logger = logger
        self.logger.info("initiated logger for stage: %s", stage)

    def on_execute(self):
        # tags that are added to each task including current task.
        tags = [
            "CONFIG-DEPOT:FAAS",
        ]

        # hints that are added to each task including current task.
        hints = [f"DEPLOY-ID:{self.Parameters.deploy_id}"]

        with self.memoize_stage["prepare_task"]:
            self.hint(hints)
            self.Parameters.tags = tags

        with self.memoize_stage[stage_name:="build_binaries"]:
            self.prepare_task_logger(stage_name)
            self.Parameters.description = "Building faas binaries"

            build_kwargs = self.Parameters.build_input_parameters

            self.logger.debug("using BILLING_FAAS_BUILD_TASK input parameters", extra={"parameters": build_kwargs})

            build_task = BillingFaasBuildTask(self, **build_kwargs)  # noqa

            build_task.Parameters.tags.extend(tags)
            build_task.hint(hints)

            build_task.save().enqueue()
            self.logger.debug("enqueued BILLING_FAAS_BUILD_TASK", extra={"task_id": build_task.id})

            self.Context.build_task_id = build_task.id
            raise sdk2.WaitTask(build_task, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])
        with self.memoize_stage[stage_name:="deploy_binaries"]:
            self.prepare_task_logger(stage_name)
            self.Parameters.description = "Deploying faas"

            build_task = sdk2.Task[self.Context.build_task_id]

            if build_task.status not in ctt.Status.Group.SUCCEED:
                raise Exception(f"Task {build_task} ended with errors")

            faas_resources = sdk2.resource.Resource.find(
                state=ctr.State.READY, task=build_task, resource_type=FaasResource
            ).limit(FAAS_RESOURCE_LIMIT)

            deploy_kwargs = self.Parameters.deploy_input_parameters

            deploy_kwargs.update(
                {
                    "faas_resources": [
                        {
                            # ci.SandboxResource has few parameters,
                            #  so DeployFaas tasklet gets data about Sandbox resources with Sandbox client,
                            #  hence it only needs id and type to handle resources.
                            # See: https://a.yandex-team.ru/arcadia/ci/tasklet/common/proto/sandbox.proto?#L11
                            "id": faas_resource.id,
                            "type": str(faas_resource.type),
                        }
                        for faas_resource in faas_resources
                    ]
                }
            )

            tasklet_kwargs = {
                "resource": self.Parameters.deploy_resource,
                "tasklet_name": "BillingDeployFaas",
                "tasklet_input_as_json": True,
                "tasklet_input": deploy_kwargs,
                "yav_token_vault": self.Parameters.tasklet_yav_token,
            }

            self.logger.debug("prepared tasklet executor kwargs: %s", tasklet_kwargs)

            tasklet_executor_task = TaskletExecutor(self, **tasklet_kwargs)

            tasklet_executor_task.Parameters.tags.extend(tags)
            tasklet_executor_task.hint(hints)

            tasklet_executor_task.save().enqueue()

            self.Context.deploy_tasklet_id = tasklet_executor_task.id
            raise sdk2.WaitTask(tasklet_executor_task.id, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])
        with self.memoize_stage[stage_name:="validate_deploy"]:
            self.prepare_task_logger(stage_name)
            self.Parameters.description = "Validating deploy tasklet status"

            tasklet_executor_task = sdk2.Task[self.Context.deploy_tasklet_id]

            self.logger.debug(f"tasklet[{tasklet_executor_task.id}] status is {tasklet_executor_task.status}")

            if tasklet_executor_task.status not in ctt.Status.Group.SUCCEED:
                raise Exception("Tasklet finished with error")

            self.Parameters.description = "Successfully deployed faas binaries"
