# -*- coding: utf-8 -*-
import logging
import os.path
import shutil

from sandbox import sdk2
from sandbox.projects.billing.resources import HotBillingManificenta
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia.sdk import mount_arc_path
from sandbox.projects.resource_types import OTHER_RESOURCE
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process


class BillingManifestsGenerationTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Запускает генерацию конфигураций Манифисентой, используя релиз из Sandbox
    (см. https://a.yandex-team.ru/arc/trunk/arcadia/billing/hot/manificenta)
    """

    class Parameters(sdk2.Task.Parameters):
        description = "billing-manificenta-manifests-generation-task"

        with sdk2.parameters.RadioGroup("Enviroment") as env:
            env.values["test"] = env.Value("test", default=True)
            env.values["dev"] = env.Value("development")
            env.values["load"] = env.Value("load")
            env.values["prod"] = env.Value("production")

        with sdk2.parameters.RadioGroup("Service") as service:
            service.values["processor"] = service.Value("processor", default=True)
            service.values["processor-yt"] = service.Value("processor-yt")
            service.values["accounts"] = service.Value("accounts")
            service.values["editor"] = service.Value("editor")
            service.values["faas"] = service.Value("faas")
            service.values["mediator"] = service.Value("mediator")

        revision = sdk2.parameters.String("Revision")
        arcadia_hash = sdk2.parameters.String("Arcadia hash")

        output_to_directory = sdk2.parameters.Bool("Output to directory", default=True)
        with output_to_directory.value["True"]:
            output_dir = sdk2.parameters.String("Output directory of manificenta", default="billing/hot/manificenta")

        resource_id = sdk2.parameters.Integer("Manificenta binary resource ID")

        with sdk2.parameters.Group("Task executor") as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        service, env = self.Parameters.service, self.Parameters.env
        self.Parameters.description = "Generate manifests for {service} with {env} env".format(service=service, env=env)

        if self.Parameters.revision:
            arcadia_url = "arcadia:/arc/trunk/arcadia@{}".format(self.Parameters.revision)
        elif self.Parameters.arcadia_hash:
            arcadia_url = "arcadia-arc:/#{}".format(self.Parameters.arcadia_hash)
        else:
            arcadia_url = "arcadia:/arc/trunk/arcadia"

        with mount_arc_path(arcadia_url) as arcadia_dir:
            logging.debug("Arcadia src dir %s", arcadia_dir)

            if self.Parameters.resource_id:
                resource = HotBillingManificenta[self.Parameters.resource_id]
            else:
                resource = HotBillingManificenta.find(attrs={"released": "stable"}).first()

            resource_path = sdk2.ResourceData(resource).path.as_posix()

            cmd = [os.path.join(arcadia_dir, resource_path), "generate", "--service", service, "--env", env]
            cmd += ["--input", "billing/hot/manificenta"]
            if self.Parameters.output_to_directory:
                cmd += ["--output", self.Parameters.output_dir]
            else:
                cmd += ["--print"]

            run_process(cmd, log_prefix="program", outputs_to_one_file=False, shell=True, work_dir=arcadia_dir)

            if self.Parameters.output_to_directory:
                origin = os.path.join(arcadia_dir, self.Parameters.output_dir)
                destination = os.path.join(os.getcwd(), self.Parameters.output_dir)
                shutil.copytree(origin, destination)
                logging.info("Create resource from %s", destination)
                task_res = channel.task.create_resource(
                    description="Manifests resource",
                    resource_path=destination,
                    resource_type=OTHER_RESOURCE,
                    attributes={"ttl": 30},
                )
                channel.task.mark_resource_ready(task_res.id)
