# -*- coding: utf-8 -*-
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.common import binary_task

from sandbox.projects.billing.tasks.Manificenta.BillingManifestsGenerationTask import BillingManifestsGenerationTask


class BillingManifestsValidationTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Запускает генерацию всех конфигураций доступных в Манифисенте
        (см. https://a.yandex-team.ru/arc/trunk/arcadia/billing/hot/manificenta)
        для проверки их корректности
    """

    class Parameters(sdk2.Task.Parameters):
        description = 'billing-manificenta-manifests-validation-task'
        with sdk2.parameters.Group('Build parameters'):
            with sdk2.parameters.CheckGroup('Environments') as envs:
                envs.values['dev'] = envs.Value('development', checked=True)
                envs.values['test'] = envs.Value('test', checked=True)
                envs.values['load'] = envs.Value('load')
                envs.values['prod'] = envs.Value('production', checked=True)

            with sdk2.parameters.CheckGroup('Services') as services:
                services.values['processor'] = services.Value('processor', checked=True)
                services.values['mediator'] = services.Value('mediator', checked=True)
                services.values['accounts'] = services.Value('accounts', checked=True)
                services.values['processor-yt'] = services.Value('processor-yt', checked=True)
                services.values['editor'] = services.Value('editor')

            arcadia_hash = sdk2.parameters.String('Arcadia hash')

            change_binary = sdk2.parameters.Bool('[Debug] Change default binary for manificenta', default=False)
            with change_binary.value['True']:
                resource_id = sdk2.parameters.Integer('Manificenta resource id')

        with sdk2.parameters.Group('Task executor') as task_executor:
            ext_params = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        generate_tasks = []

    def create_generate_task(self, env, service):
        generate_kwargs = {'env': env, 'service': service, 'output_to_directory': False}
        if self.Parameters.arcadia_hash:
            generate_kwargs['arcadia_hash'] = self.Parameters.arcadia_hash
        if self.Parameters.change_binary:
            generate_kwargs['resource_id'] = self.Parameters.resource_id

        return BillingManifestsGenerationTask(self, **generate_kwargs).save().enqueue()

    def generate(self, envs, services):
        self.Parameters.description = 'Generate manifests'
        tasks = []
        for env in envs:
            tasks.extend([self.create_generate_task(env, service) for service in services])
        self.Context.generate_tasks = [task.id for task in tasks]
        raise sdk2.WaitTask(tasks, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])

    def validate(self, task_ids):
        self.Parameters.description = 'Validate statuses'
        for task_id in task_ids:
            task = sdk2.Task[task_id]
            if task.status not in ctt.Status.Group.SUCCEED:
                raise Exception('Could not finish running manificenta task: {id}'.format(id=task.id))

        self.Parameters.description = 'Validation completed.'

    def on_execute(self):
        envs, services = self.Parameters.envs, self.Parameters.services
        with self.memoize_stage['generate']:
            self.generate(envs, services)

        with self.memoize_stage['validate']:
            self.validate(self.Context.generate_tasks)
