# coding: utf-8

import os
import logging
from sandbox import sdk2
import re
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.projects.juggler.BuildJugglerChecksBundle
import sandbox.projects.bitbucket.common.git_helpers as git


class GitRefIdParameter(SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class GitRepoUrlParameter(SandboxStringParameter):
    name = 'repo_url'
    description = 'Git Repo URL'
    required = True
    default_value = ''


class VaultSecretOwner(SandboxStringParameter):
    name = 'vault_secret_owner'
    description = 'Vault Secret Owner'
    required = False
    default_value = ''


class VaultSecretName(SandboxStringParameter):
    name = 'vault_secret_name'
    description = 'Vault Secret name'
    required = False
    default_value = 'bitbucket_token'


class VaultSecretUser(SandboxStringParameter):
    name = 'vault_secret_user'
    description = 'Username to use with vault secret'
    required = False
    default_value = 'x-oauth-token'


class BuildJugglerChecksBundleGit(sandbox.projects.juggler.BuildJugglerChecksBundle.BuildJugglerChecksBundle):
    """
    Create new-style checks bundle
    """
    execution_space = 5120

    type = 'BUILD_JUGGLER_CHECKS_BUNDLE_GIT'

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        GitRepoUrlParameter
    ]

    def __init__(self, task_id=0):
        sandbox.projects.juggler.BuildJugglerChecksBundle.BuildJugglerChecksBundle.__init__(self, task_id)

    def on_execute(self):
        self.__fill_missing_parameters()
        sandbox.projects.juggler.BuildJugglerChecksBundle.BuildJugglerChecksBundle.on_execute(self)

    def __fill_missing_parameters(self):
        repo_url = self.ctx.get(GitRepoUrlParameter.name)
        if not repo_url:
            raise SandboxTaskFailureError("You need to pass repo_url to this task")
        self.ctx[sandbox.projects.juggler.BuildJugglerChecksBundle.BundleNameParameter.name] = BuildJugglerChecksBundleGit.__get_bundle_name_from_url(repo_url)
        self.ctx[sandbox.projects.juggler.BuildJugglerChecksBundle.ResourceTypeParameter.name] = sandbox.projects.juggler.BuildJugglerChecksBundle.ResourceTypeParameter.default_value
        self.ctx[sandbox.projects.juggler.BuildJugglerChecksBundle.UploadToMdsParameter.name] = False

    def _get_package_path(self):
        if not self.ctx.get(GitRepoUrlParameter.name):
            raise SandboxTaskFailureError("You need to pass repo_url to this task")
        vault_username = self.ctx.get(VaultSecretUser.name)
        if not vault_username:
            vault_username = 'x-oauth-token'
        vault_owner = self.ctx.get(VaultSecretOwner.name)
        if not vault_owner or vault_owner == '':
            vault_owner = self.ctx.get('owner')
        vault_secret_name = self.ctx.get(VaultSecretName.name)
        if not vault_secret_name:
            vault_secret_name = 'bitbucket_token'
        vault_password = self.get_vault_data(vault_owner, vault_secret_name)
        repo_url = git.get_repo_url_with_credentials(
            repo_url=self.ctx.get(GitRepoUrlParameter.name),
            user=vault_username,
            password=vault_password
        )
        path = 'git_bundle'
        git.git_clone(self, repo_url, path)
        git.git_fetch(self, path)
        git.git_checkout(self,
                         path,
                         self.ctx.get(GitRefShaParameter.name) or self.ctx.get(GitRefIdParameter.name)
                         )
        bundle = BuildJugglerChecksBundleGit.build_bundle(self, path)
        return bundle

    @staticmethod
    def __get_bundle_name_from_url(repo_url):
        parser = re.compile('^.*/scm/(?P<project>.*)/(?P<repo>.*).git')
        match = parser.match(repo_url)
        if not match:
            raise SandboxTaskFailureError("Could not determine repo name")
        return '{}-{}'.format(match.groupdict()['project'], match.groupdict()['repo'])

    @staticmethod
    def build_bundle(task, repo_dir):
        chmod_command = 'chmod +x *.*'
        build_command = 'tar cvf ../bundle.tar *'
        cleanup_command = 'rm -Rf {}'.format(repo_dir)
        with sdk2.helpers.ProcessLog(task=task, logger=logging.getLogger('build_bundle')) as pl:
            if os.path.exists(repo_dir):
                rc = sdk2.helpers.subprocess.Popen(
                    chmod_command,
                    shell=True,
                    cwd=repo_dir,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                ).wait()
                if rc != 0:
                    raise SandboxTaskFailureError("chmod failed")
                rc = sdk2.helpers.subprocess.Popen(
                    build_command,
                    shell=True,
                    cwd=repo_dir,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                ).wait()
                if rc != 0:
                    raise SandboxTaskFailureError("tar failed")
                rc = sdk2.helpers.subprocess.Popen(
                    cleanup_command.split(),
                    stderr=pl.stderr,
                    stdout=pl.stdout
                ).wait()
                if rc != 0:
                    raise SandboxTaskFailureError("rm -Rf failed")
        return os.path.abspath('bundle.tar')
