# coding: utf-8

import logging
import os
import yaml

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.bitbucket.common.git_task_parameters import GitParameters
from sandbox.projects.sandbox.resources import LXC_CONTAINER
import sandbox.common.types.misc as ctm
import sandbox.projects.bitbucket.common.git_helpers as git


class GitTarballPkg(sdk2.Resource):
    """Tarball from git scripts"""
    share = True
    releasable = True


class BuildPkgFromGit(sdk2.Task):
    GIT_DIR = 'git_bundle'
    BUILD_META_FILE = 'build/sandbox/build.yaml'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024
        # Required for containers with docker, because most docker-registries are ipv4 only
        dns = ctm.DnsType.DNS64

    class Parameters(GitParameters):
        build_script = sdk2.parameters.String('Build Script',
                                              description='build script executed instead <repo>/build/sandbox/build.sh',
                                              default='',
                                              required=False
                                              )
        container = sdk2.parameters.Container(
            "LXC Container resource to use",
            resource_type=LXC_CONTAINER,
            required=False,
            default=None
        )
        privileged = sdk2.parameters.Bool("You will be root inside LXC")
        ramdrive = sdk2.parameters.Integer("Create a RAM drive of specified size in GiB", default=32)

    def on_save(self):
        super(BuildPkgFromGit, self).on_save()
        self.Requirements.privileged = self.Parameters.privileged
        if self.Parameters.ramdrive:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Parameters.ramdrive << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))
        else:
            logging.info('Ramdrive disabled')

    def on_execute(self):
        vault_secret_owner = self.Parameters.vault_secret_owner
        if vault_secret_owner == "":
            vault_secret_owner = self.owner
        repo_url = git.get_repo_url_with_credentials(self.Parameters.repo_url, self.Parameters.vault_secret_user,
                                                     sdk2.Vault.data(vault_secret_owner,
                                                                     self.Parameters.vault_secret_name))
        self.setup_ramdrive()

        work_dir = BuildPkgFromGit.GIT_DIR
        git.git_init(self, work_dir)
        git.git_fetch(self, work_dir, repo_url, self.Parameters.ref_sha or self.Parameters.ref_id, depth=1)
        git.git_checkout(self, work_dir, 'FETCH_HEAD')

        build_meta_file = '{}/{}'.format(work_dir, BuildPkgFromGit.BUILD_META_FILE)
        if not os.path.isfile(build_meta_file):
            raise TaskFailure('No {} present in repo'.format(BuildPkgFromGit.BUILD_META_FILE))
        try:
            with open(build_meta_file) as f:
                metadata = yaml.load(f)
        except Exception as error:
            raise TaskFailure('Cannot parse build metadata file: {}'.format(error))
        build_script = '{}/{}'.format(work_dir, metadata['build_cmd'])
        if os.path.isfile(build_script):
            os.chmod(build_script, 0o755)
        else:
            raise TaskFailure('build_cmd script absent')

        env = os.environ.copy()
        env['BUILD_RESOURCE_PATH'] = str(self.path())
        with sdk2.helpers.ProcessLog(task=self, logger="build") as pl:
            proc = sdk2.helpers.subprocess.Popen(
                metadata['build_cmd'],
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                cwd=work_dir,
                env=env)

            url = "%s/%s" % (self.log_resource.http_proxy,
                             pl.stdout.path.relative_to(self.path(self.log_resource.path)))
            tail_sfx = '?tail=1&force_text_mode=1'
            self.set_info('Process started. <a href="{0}" target="_blank">output</a>'.format(url), do_escape=False)
            self.set_info('Process <a href="{0}{1}">tail log </a>'.format(url, tail_sfx), do_escape=False)
            logging.info('wait task')
            proc.wait()

            logging.error('Script err_code:%d', proc.returncode)
            if proc.returncode != 0:
                raise TaskFailure('build_cmd has non-zero exit status')

        resource = GitTarballPkg(
            self,
            description=metadata.get('description') or 'git tarball package',
            path=metadata.get('resource_archive') or 'resource.tar.xz'
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
