# coding: utf-8

import re
from sandbox import sdk2
from sandbox import common
from sandbox.common.errors import TaskFailure
from sandbox.projects.bitbucket.common.git_task_parameters import GitParameters
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.common.infra.build_deb_package import DebPackage
from sandbox.projects import resource_types


class FetchBuildTest(sdk2.Task):
    SUBTASK_TIMEOUT = 60 * 60  # 60 min
    RESOURCE_NUMB_LIMIT = 32

    TEST_THREADS = 6

    CORES_REQ = 32
    RAM_REQ = 128 << 10  # 128 GB
    DISK_REQ = 20 << 10  # 20 GB

    LINUX_IMAGE_SERVER_RESOURCE_ID = None
    LINUX_IMAGE_EXTRA_RESOURCE_ID = None
    LINUX_TOOLS_RESOURCE_ID = None
    LINUX_IMAGE_RESOURCE_ID = None
    LINUX_HEADERS_RESOURCE_ID = None

    LINUX_IMAGE_SERVER_FILENAME = None
    LINUX_IMAGE_EXTRA_FILENAME = None
    LINUX_TOOLS_FILENAME = None
    LINUX_IMAGE_FILENAME = None
    LINUX_HEADERS_FILENAME = None

    class Requirements(sdk2.Task.Requirements):
        # Required for containers with docker, because most docker-registries are ipv4 only
        dns = ctm.DnsType.DNS64

    class Parameters(GitParameters):
        with sdk2.parameters.Group("BuildEnv"):
            platform = sdk2.parameters.String(
                "Target platform",
                choices=[('linux_ubuntu_18.04_bionic', 'linux_ubuntu_18.04_bionic'),
                         ('linux_ubuntu_16.04_xenial', 'linux_ubuntu_16.04_xenial'),
                         ('linux_ubuntu_14.04_trusty', 'linux_ubuntu_14.04_trusty'),
                         ('linux_ubuntu_12.04_precise', 'linux_ubuntu_12.04_precise')],
                default='linux_ubuntu_16.04_xenial',
                required=True
            )

    def unparse_ref_id(self):
        ref_id = self.Parameters.ref_id
        official_build = False

        patterns = [
            # refs/heads/yandex-X.Y
            r'^refs/heads/yandex-[0-9]+\.[0-9]+$',
            # refs/tags/vX.Y.Z
            r'^refs/tags/v[0-9]+\.[0-9]+\.[0-9]+$',
            # refs/tags/vX.Y.Z-A
            r'^refs/tags/v[0-9]+\.[0-9]+\.[0-9]+-[0-9]+$',
            # refs/tags/vX.Y.Z-A.B
            r'^refs/tags/v[0-9]+\.[0-9]+\.[0-9]+-[0-9]+\.[0-9]+$',
            # refs/tags/vX.Y.Z-A.*
            r'^refs/tags/v[0-9]+\.[0-9]+\.[0-9]+-[0-9]+\..*$',
        ]

        if ref_id:
            for p in patterns:
                if re.match(p, ref_id) is not None:
                    official_build = True
                    break

        if not official_build:
            suffix = '.dev-{revision}-{task_id}'
            notice = 'Autobuild triggered by PR'
        else:
            suffix = ''
            notice = ''

        return suffix, notice

    def on_execute(self):
        vault_secret_owner = self.Parameters.vault_secret_owner
        if vault_secret_owner == "":
            vault_secret_owner = self.owner

        suffix, notice = self.unparse_ref_id()

        with self.memoize_stage.build:
            build_task_class = sdk2.Task['PBUILDER_BUILD_DEB_PACKAGE']
            build_task = build_task_class(
                self,
                src_origin='git',
                git_url=self.Parameters.repo_url,
                git_head=self.Parameters.ref_sha or self.Parameters.ref_id,
                git_shallow=True,
                git_token='{}:{}'.format(self.owner, self.Parameters.vault_secret_name),
                platform=self.Parameters.platform,
                # distribution="xenial"
                config_url="arcadia:/arc/trunk/arcadia/infra/kernel/scripts/pbuilderrc",
                build_parallel=True,
                pbuilder_tarball_origin="pbuilderrc",
                gpg_public="robot-rtc-builder-gpg-public",
                gpg_private="robot-rtc-builder-gpg-private",
                release_dist_repo="search-kernel",
                release_upload_packages=True,
                release_subscribers="yakernel-team@yandex-team.ru",
                save_packages=True,
                sign_packages=True,
                ssh_login="robot-rtc-builder",
                ssh_private="robot-rtc-builder-ssh",
                ramdrive_size=8192,
                changelog_embed_suffix=suffix,
                changelog_embed_notice=notice
            )

            build_task.Requirements.cores = self.CORES_REQ
            build_task.Requirements.ram = self.RAM_REQ
            build_task.Requirements.disk_space = self.DISK_REQ

            self.Context.build_task_id = build_task.save().enqueue().id
            self.set_info("Stage [BUILD]: start <a href='https://sandbox.yandex-team.ru/task/{}/resources'>task</a>".format(self.Context.build_task_id),
                          do_escape=False)
            raise sdk2.WaitTask(self.Context.build_task_id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, timeout=FetchBuildTest.SUBTASK_TIMEOUT)

        with self.memoize_stage.check_build_status:
            build_task = sdk2.Task[self.Context.build_task_id]
            self.set_info("Stage [BUILD]: finished with status: {}".format(build_task.status))

            if build_task.status not in ctt.Status.Group.SUCCEED:
                if build_task.status not in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
                    build_task.stop()
                    raise TaskFailure('Build subtask exceeded timeout of {} seconds'.format(FetchBuildTest.SUBTASK_TIMEOUT))
                else:
                    raise TaskFailure('Build subtask failed')

        with self.memoize_stage.test:
            task_resources = sdk2.Resource.find(task_id=self.Context.build_task_id, type=DebPackage, state=ctr.State.READY).limit(self.RESOURCE_NUMB_LIMIT)

            for r in task_resources:
                if (self.LINUX_IMAGE_SERVER_RESOURCE_ID is None) and (r.deb_package == 'linux-image-server'):
                    self.LINUX_IMAGE_SERVER_RESOURCE_ID = r.id
                    self.LINUX_IMAGE_SERVER_RESOURCE_NAME = r.description

                if (self.LINUX_IMAGE_EXTRA_RESOURCE_ID is None) and (r.deb_package == 'linux-image-extra-' + r.deb_version):
                    self.LINUX_IMAGE_EXTRA_RESOURCE_ID = r.id
                    self.LINUX_IMAGE_EXTRA_RESOURCE_NAME = r.description

                if (self.LINUX_TOOLS_RESOURCE_ID is None) and (r.deb_package == 'linux-tools'):
                    self.LINUX_TOOLS_RESOURCE_ID = r.id
                    self.LINUX_TOOLS_RESOURCE_NAME = r.description

                if (self.LINUX_IMAGE_RESOURCE_ID is None) and (r.deb_package == 'linux-image-' + r.deb_version):
                    self.LINUX_IMAGE_RESOURCE_ID = r.id
                    self.LINUX_IMAGE_RESOURCE_NAME = r.description

                if (self.LINUX_HEADERS_RESOURCE_ID is None) and (r.deb_package.startswith('linux-headers')):
                    self.LINUX_HEADERS_RESOURCE_ID = r.id
                    self.LINUX_HEADERS_RESOURCE_NAME = r.description

            if None in (self.LINUX_IMAGE_SERVER_RESOURCE_ID, self.LINUX_IMAGE_EXTRA_RESOURCE_ID, self.LINUX_TOOLS_RESOURCE_ID, self.LINUX_IMAGE_RESOURCE_ID, self.LINUX_HEADERS_RESOURCE_ID):
                raise TaskFailure('Resource(s) not found:{}{}{}{}{}'.format(
                    ' linux-image-server' if self.LINUX_IMAGE_SERVER_RESOURCE_ID is None else '',
                    ' linux-image-extra' if self.LINUX_IMAGE_EXTRA_RESOURCE_ID is None else '',
                    ' linux-tools' if self.LINUX_TOOLS_RESOURCE_ID is None else '',
                    ' linux-image' if self.LINUX_IMAGE_RESOURCE_ID is None else '',
                    ' linux-headers' if self.LINUX_HEADERS_RESOURCE_ID is None else ''))

            test_task_class = sdk2.Task['YA_MAKE_TGZ']
            test_task = test_task_class(
                self,
                arts='infra/environments/rtc-xenial/vm-layer-unstable/layer.tar.zst; '
                     'infra/environments/rtc-xenial/vm-image-unstable/rootfs.img; '
                     'infra/kernel/ci/ut/rtc-xenial-unstable/test-results',
                definition_flags=('-DKERN_TEST=1 '
                                  '-DKERNEL_TEST_SKIP_PERF=1 '
                                  '-DLINUX_IMAGE_SERVER_RESOURCE={} '
                                  '-DLINUX_IMAGE_EXTRA_RESOURCE={} '
                                  '-DLINUX_TOOLS_RESOURCE={} '
                                  '-DLINUX_IMAGE_RESOURCE={} '
                                  '-DLINUX_HEADERS_RESOURCE={} '
                                  '-DLINUX_IMAGE_SERVER_FILENAME={} '
                                  '-DLINUX_IMAGE_EXTRA_FILENAME={} '
                                  '-DLINUX_TOOLS_FILENAME={} '
                                  '-DLINUX_IMAGE_FILENAME={} '
                                  '-DLINUX_HEADERS_FILENAME={}').format(
                    self.LINUX_IMAGE_SERVER_RESOURCE_ID,
                    self.LINUX_IMAGE_EXTRA_RESOURCE_ID,
                    self.LINUX_TOOLS_RESOURCE_ID,
                    self.LINUX_IMAGE_RESOURCE_ID,
                    self.LINUX_HEADERS_RESOURCE_ID,
                    self.LINUX_IMAGE_SERVER_RESOURCE_NAME,
                    self.LINUX_IMAGE_EXTRA_RESOURCE_NAME,
                    self.LINUX_TOOLS_RESOURCE_NAME,
                    self.LINUX_IMAGE_RESOURCE_NAME,
                    self.LINUX_HEADERS_RESOURCE_NAME),
                targets='infra/environments/rtc-xenial/vm-layer-unstable; '
                        'infra/environments/rtc-xenial/vm-image-unstable; '
                        'infra/kernel/ci/ut/rtc-xenial-unstable',
                test=True,
                test_threads=self.TEST_THREADS,
                junit_report=True,
                use_aapi_fuse=True,
                resource_archive_check=True,
            )

            test_task.Requirements.cores = self.CORES_REQ
            test_task.Requirements.ram = self.RAM_REQ
            test_task.Requirements.disk_space = 2 * self.DISK_REQ  # 40 GB, same as default

            subtask_platform_tag = common.platform.PLATFORM_TO_TAG.get(self.Parameters.platform)
            test_task.Requirements.client_tags &= subtask_platform_tag

            self.Context.test_task_id = test_task.save().enqueue().id
            self.set_info("Stage [TEST]: start <a href='https://sandbox.yandex-team.ru/task/{}/view'>task</a>".format(self.Context.test_task_id),
                          do_escape=False)
            raise sdk2.WaitTask(self.Context.test_task_id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, timeout=FetchBuildTest.SUBTASK_TIMEOUT)

        with self.memoize_stage.check_test_status:
            test_task = sdk2.Task[self.Context.test_task_id]
            self.set_info("Stage [TEST]: finished with status: {}".format(test_task.status))
            junit_reports = sdk2.Resource.find(task_id=self.Context.test_task_id,
                                               type=resource_types.JUNIT_REPORT).limit(self.RESOURCE_NUMB_LIMIT)
            for r in junit_reports:
                self.set_info("Stage [TEST]: reports <a href='{}'>{}</a>".format(r.http_proxy, r.description), do_escape=False)
            if test_task.status not in ctt.Status.Group.SUCCEED:
                if test_task.status not in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
                    test_task.stop()
                    raise TaskFailure('Test subtask exceeded timeout of {} seconds'.format(FetchBuildTest.SUBTASK_TIMEOUT))
                else:
                    raise TaskFailure('Test subtask failed')
