# coding: utf-8

import logging
import os
import yaml

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.bitbucket.common.git_task_parameters import GitParameters
from sandbox.projects.sandbox.resources import LXC_CONTAINER
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.projects.bitbucket.common.git_helpers as git


class FetchFromBbAndTestInArc(sdk2.Task):
    GIT_DIR = 'git_bundle'
    TASK_PARAMS_YAML = 'ci/sandbox_task.yaml'
    SUBTASK_TIMEOUT = 2 * 60 * 60  # 2 h

    DC_TO_TAG = {
        "SAS": ctc.Tag.SAS,
        "VLA": ctc.Tag.VLA,
        "MAN": ctc.Tag.MAN,
        "IVA": ctc.Tag.IVA,
        "MYT": ctc.Tag.MYT,
    }

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 5 * 1024
        cores = 1
        cpu = 1
        ram = 1024

        # Required for containers with docker, because most docker-registries are ipv4 only
        dns = ctm.DnsType.DNS64

        # do not use shared caches
        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(GitParameters):
        task_params_yaml = sdk2.parameters.String('Task Parameters YAML',
                                                  description='custom yaml file with task parameters instead of <repo>/ci/sandbox_task.yaml',
                                                  default='',
                                                  required=False
                                                  )

        with sdk2.parameters.Group("BuildEnv"):
            platform = sdk2.parameters.String(
                "Target platform",
                choices=[('linux_ubuntu_18.04_bionic', 'linux_ubuntu_18.04_bionic'),
                         ('linux_ubuntu_16.04_xenial', 'linux_ubuntu_16.04_xenial'),
                         ('linux_ubuntu_14.04_trusty', 'linux_ubuntu_14.04_trusty'),
                         ('linux_ubuntu_12.04_precise', 'linux_ubuntu_12.04_precise')],
                default='linux_ubuntu_16.04_xenial',
                required=True
            )
            container = sdk2.parameters.Container(
                "LXC Container resource to use",
                description="If not set, use latest release of LXC_CONTAINER for target platform",
                resource_type=LXC_CONTAINER,
                required=False,
                default=None,
            )

        privileged = sdk2.parameters.Bool("You will be root inside LXC")
        ramdrive = sdk2.parameters.Integer("Create a RAM drive of specified size in GiB", default=32)

    def on_enqueue(self):
        if self.Parameters.container is None:
            self.Parameters.container = LXC_CONTAINER.find(
                attrs={"platform": self.Parameters.platform,
                       "released": "stable"}).first()

    def on_save(self):
        super(FetchFromBbAndTestInArc, self).on_save()
        self.Requirements.privileged = self.Parameters.privileged
        if self.Parameters.ramdrive:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Parameters.ramdrive << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))
        else:
            logging.info('Ramdrive disabled')

    def task_set_requirements(self, task, reqs):
        task.Requirements.client_tags = ~ctc.Tag.MULTISLOT & ctc.Tag.SSD & ctc.Tag.Group.LINUX & ctc.Tag.GENERIC
        if reqs is None:
            return
        cores = reqs.get('cores')
        if cores is not None:
            task.Requirements.cores = cores

        ram = reqs.get('ram')
        if ram is not None:
            task.Requirements.ram = ram

        platform = reqs.get('platform')
        dc = reqs.get('dc')

        subtask_platform_tag = common.platform.PLATFORM_TO_TAG.get(platform or self.Parameters.platform)
        task.Requirements.client_tags &= subtask_platform_tag

        if dc is not None:
            subtask_dc_tag = self.DC_TO_TAG.get(dc.upper())
            task.Requirements.client_tags &= subtask_dc_tag

    def parse_v0_config(self, task_params):

        sandbox_task = task_params.get('sandbox_task')
        if sandbox_task is None:
            raise TaskFailure('No subtask to start')

        sandbox_task_type = sandbox_task.get('type')
        if sandbox_task_type is None:
            raise TaskFailure('No subtask type is provided')

        if sandbox_task_type == 'YA_MAKE_TGZ':
            if 'definition_flags' not in sandbox_task:
                sandbox_task['definition_flags'] = "-DCOMMIT_SHA={}".format(self.Parameters.ref_sha)

            if 'env_vars' not in sandbox_task:
                sandbox_task['env_vars'] = "OAUTH_TOKEN=$(vault:value:{}:{})".format(self.Context.vault_secret_owner, self.Parameters.vault_secret_name)

        ymake_tgz_class = sdk2.Task[sandbox_task_type]
        ymake_tgz_subtask = ymake_tgz_class(
            self,
            **sandbox_task
        )

        tsk_req = task_params.get('requirements')
        # Convert ram requirement to bytes
        req_ram = tsk_req.get('ram')
        if req_ram is not None:
            tsk_req['ram'] = req_ram << 10

        self.task_set_requirements(ymake_tgz_subtask, tsk_req)
        return ymake_tgz_subtask

    def parse_v1_config(self, task_params):
        subtask_list = []
        subtasks = task_params.get('jobs', [])

        for subtask in subtasks:
            subtask_list.append(self.parse_v0_config(subtasks[subtask]))
        return subtask_list

    def on_execute(self):
        self.Context.vault_secret_owner = self.Parameters.vault_secret_owner
        if self.Context.vault_secret_owner == "":
            self.Context.vault_secret_owner = self.owner

        with self.memoize_stage.fetch_from_bb:
            repo_url = git.get_repo_url_with_credentials(self.Parameters.repo_url, self.Parameters.vault_secret_user,
                                                         sdk2.Vault.data(self.Context.vault_secret_owner,
                                                                         self.Parameters.vault_secret_name))
            self.setup_ramdrive()

            work_dir = FetchFromBbAndTestInArc.GIT_DIR
            git.git_init(self, work_dir)
            git.git_fetch(self, work_dir, repo_url, self.Parameters.ref_sha or self.Parameters.ref_id, depth=1)
            git.git_checkout(self, work_dir, 'FETCH_HEAD')

            yaml_file = self.Parameters.task_params_yaml if self.Parameters.task_params_yaml != '' else FetchFromBbAndTestInArc.TASK_PARAMS_YAML
            yaml_file_path = '{}/{}'.format(work_dir, yaml_file)
            if not os.path.isfile(yaml_file_path):
                raise TaskFailure('No {} present in repo'.format(yaml_file))
            try:
                with open(yaml_file_path) as f:
                    self.Context.task_params = yaml.load(f)
            except Exception as error:
                raise TaskFailure('Cannot parse task_params yaml: {}'.format(error))

        with self.memoize_stage.test_in_arc:
            kind = self.Context.task_params.get('kind')
            if kind is None:
                subtasks = [self.parse_v0_config(self.Context.task_params)]
            elif kind == "gbgadaptor/v1":
                subtasks = self.parse_v1_config(self.Context.task_params)

            self.Context.subtask_list = []
            for tsk in subtasks:
                self.Context.subtask_list.append(tsk.save().enqueue().id)
            raise sdk2.WaitTask(self.Context.subtask_list, ctt.Status.Group.FINISH, wait_all=True, timeout=FetchFromBbAndTestInArc.SUBTASK_TIMEOUT)

        with self.memoize_stage.check_subtask_status:
            for tsk_id in self.Context.subtask_list:
                subtask = sdk2.Task[tsk_id]
                if subtask.status not in ctt.Status.Group.SUCCEED:
                    raise TaskFailure('Subtask id: {} failed with status: {}'.format(tsk_id, subtask.status))
