# coding: utf-8

import logging
import os
import yaml
import jinja2
import re

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.bitbucket.common.git_task_parameters import GitParameters
from sandbox.projects.sandbox.resources import LXC_CONTAINER
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.projects.bitbucket.common.git_helpers as git


class GBGAdaptor(sdk2.Task):
    "Generic go-baby-go adaptor task"

    GIT_DIR = 'git_bundle'
    TASK_PARAMS_YAML = 'ci/gbgadaptor.yaml'
    SUBTASK_TIMEOUT = 30 * 60  # 30 min

    DC_TO_TAG = {
        "SAS": ctc.Tag.SAS,
        "VLA": ctc.Tag.VLA,
        "MAN": ctc.Tag.MAN,
        "IVA": ctc.Tag.IVA,
        "MYT": ctc.Tag.MYT,
    }

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 5 * 1024
        cores = 1
        cpu = 1
        ram = 1024

        # Required for containers with docker, because most docker-registries are ipv4 only
        dns = ctm.DnsType.DNS64

        # do not use shared caches
        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(GitParameters):
        task_params_yaml = sdk2.parameters.String('Task Parameters YAML',
                                                  description='custom yaml file with task parameters instead of <repo>/ci/gbgadaptor.yaml',
                                                  default='',
                                                  required=False
                                                  )

        with sdk2.parameters.Group("BuildEnv"):
            platform = sdk2.parameters.String(
                "Target platform",
                choices=[('linux_ubuntu_18.04_bionic', 'linux_ubuntu_18.04_bionic'),
                         ('linux_ubuntu_16.04_xenial', 'linux_ubuntu_16.04_xenial'),
                         ('linux_ubuntu_14.04_trusty', 'linux_ubuntu_14.04_trusty'),
                         ('linux_ubuntu_12.04_precise', 'linux_ubuntu_12.04_precise')],
                default='linux_ubuntu_16.04_xenial',
                required=True
            )
            container = sdk2.parameters.Container(
                "LXC Container resource to use",
                description="If not set, use latest release of LXC_CONTAINER for target platform",
                resource_type=LXC_CONTAINER,
                required=False,
                default=None,
            )

        privileged = sdk2.parameters.Bool("You will be root inside LXC")
        ramdrive = sdk2.parameters.Integer("Create a RAM drive of specified size in GiB", default=32)

    def on_enqueue(self):
        if self.Parameters.container is None:
            self.Parameters.container = LXC_CONTAINER.find(
                attrs={"platform": self.Parameters.platform,
                       "released": "stable"}).first()

    def on_save(self):
        super(GBGAdaptor, self).on_save()
        self.Requirements.privileged = self.Parameters.privileged
        if self.Parameters.ramdrive:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Parameters.ramdrive << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))
        else:
            logging.info('Ramdrive disabled')

    def render_one_task(self, params, template):
        task = dict(template)
        logging.debug("render_args: {}".format(params))
        logging.debug("task_template: {}".format(task))
        for k, v in task.iteritems():
            if type(v) != str:
                continue
            try:
                task[k] = jinja2.Template(v).render(params)
            except:
                raise TaskFailure("Fail to render key:{} val:{}".format(k, v))
            logging.debug("task_rendered: {}".format(task))
        return task

    def task_set_requirements(self, task, reqs):
        task.Requirements.client_tags = ctc.Tag.SSD & ctc.Tag.LINUX_XENIAL & ctc.Tag.GENERIC
        if reqs is None:
            return
        cores = reqs.get('cores')
        if cores is not None:
            task.Requirements.cores = cores

        ram = reqs.get('ram')
        if ram is not None:
            task.Requirements.ram = ram

    def parse_v0_config(self, task_params):
        subtask_list = []
        tsk_opts = task_params.get('sandbox_task')
        if tsk_opts is None:
            raise TaskFailure('No subtask to start')

        tsk_type = tsk_opts.get('type')
        if tsk_type is None:
            raise TaskFailure('No subtask type is provided')
        tsk_opts = self.render_one_task(self.render_params, tsk_opts)
        sbtask_class = sdk2.Task[tsk_type]
        tsk_req = task_params.get('requirements')
        subtask = sbtask_class(
            self,
            **tsk_opts
        )
        # Convert ram requirement to bytes
        req_ram = tsk_req.get('ram')
        if req_ram is not None:
            tsk_req['ram'] = req_ram << 10
        self.task_set_requirements(subtask, tsk_req)
        subtask_list.append(subtask)
        return subtask_list

    def parse_v1_config(self, task_params):
        subtask_list = []
        items = task_params.get('jobs', [])

        for name in items:
            logging.debug("walk task-group name: {}, body: {}".format(name, items[name]))

            ref_pattern = items[name].get('git_ref_pattern')
            if ref_pattern is not None:
                m = re.search(ref_pattern, self.Parameters.ref_id)
                if m is None:
                    logging.info("Ignore task-group: {}, ref_pattern:{}, ref_id: {}".format(name, ref_pattern, self.Parameters.ref_id))
                    continue

            tasks = items[name].get('sandbox_tasks')
            if tasks is None:
                raise TaskFailure('Task list is empty for task-group: {}'.format(name))
            for t in tasks:
                tsk_type = t.get('type')
                tsk_opts = t.get('custom_fields', {})
                tsk_req = t.get('requirements', {})
                logging.debug("handle task type:{}, custom_fields:{} requirements:{}".format(tsk_type, tsk_opts, tsk_req))
                if tsk_type is None:
                    raise TaskFailure('No task type is provided, for task spec: {}, group: {}'.format(t, name))
                sbtask_class = sdk2.Task[tsk_type]
                tsk_opts = self.render_one_task(self.render_params, tsk_opts)
                subtask = sbtask_class(
                    self,
                    **tsk_opts
                )
                self.task_set_requirements(subtask, tsk_req)
                subtask_list.append(subtask)
        return subtask_list

    def on_execute(self):
        self.Context.vault_secret_owner = self.Parameters.vault_secret_owner
        if self.Context.vault_secret_owner == "":
            self.Context.vault_secret_owner = self.owner
        with self.memoize_stage.fetch_from_bb:
            repo_url = git.get_repo_url_with_credentials(self.Parameters.repo_url, self.Parameters.vault_secret_user,
                                                         sdk2.Vault.data(self.Context.vault_secret_owner,
                                                                         self.Parameters.vault_secret_name))
            self.setup_ramdrive()

            work_dir = GBGAdaptor.GIT_DIR
            git.git_init(self, work_dir)
            git.git_fetch(self, work_dir, repo_url, self.Parameters.ref_sha or self.Parameters.ref_id, depth=1)
            git.git_checkout(self, work_dir, 'FETCH_HEAD')

            yaml_file = self.Parameters.task_params_yaml if self.Parameters.task_params_yaml != '' else GBGAdaptor.TASK_PARAMS_YAML
            yaml_file_path = '{}/{}'.format(work_dir, yaml_file)
            if not os.path.isfile(yaml_file_path):
                self.set_info('No {} present in repo, exit.'.format(yaml_file))
                return
            try:
                with open(yaml_file_path) as f:
                    self.Context.task_params = yaml.load(f)
                    logging.debug("task_params: {}".format(self.Context.task_params))
            except Exception as error:
                raise TaskFailure('Cannot parse task_params yaml: {}'.format(error))

        with self.memoize_stage.run_subtask:
            bb_param = {}
            bb_param["ref_id"] = self.Parameters.ref_id
            bb_param["ref_sha"] = self.Parameters.ref_sha
            bb_param["repo_url"] = self.Parameters.repo_url
            bb_param["vault_secret_owner"] = self.Context.vault_secret_owner
            bb_param["vault_secret_name"] = self.Parameters.vault_secret_name
            bb_param["vault_secret_user"] = self.Parameters.vault_secret_user
            self.render_params = {"bitbucket": bb_param}

            kind = self.Context.task_params.get('kind')
            if kind is None:
                subtasks = self.parse_v0_config(self.Context.task_params)
            elif kind == "gbgadaptor/v1":
                subtasks = self.parse_v1_config(self.Context.task_params)
            logging.debug("subtask_list: {}".format(subtasks))

            self.Context.subtask_list = []
            for tsk in subtasks:
                self.Context.subtask_list.append(tsk.save().enqueue().id)
            logging.debug("wait subtasks {}".format(self.Context.subtask_list))
            raise sdk2.WaitTask(self.Context.subtask_list, ctt.Status.Group.FINISH, wait_all=True)

        with self.memoize_stage.check_subtask_status:
            ex = None
            for tsk_id in self.Context.subtask_list:
                subtask = sdk2.Task[tsk_id]
                if subtask.status not in ctt.Status.Group.SUCCEED:
                    raise TaskFailure('Subtask id: {} failed with status: {}'.format(tsk_id, subtask.status))

    def on_release(self, parameters):
        for tsk_id in self.Context.subtask_list:
            subtask = sdk2.Task[tsk_id]
            self.set_info('Create release for subtask {}'.format(subtask.id))
            param = dict(parameters)
            param['task_id']
            self.server.release(
                task_id=tsk_id,
                type=parameters['release_status'],
                subject=parameters['release_status']
            )
