# coding: utf-8

import re
import os
import logging
import time
from sandbox import sdk2
from sandbox.common.errors import TaskFailure


def get_repo_url_with_credentials(repo_url, user, password):
    parser = re.compile('^https://(?P<address_part>.*)$')
    match = parser.match(repo_url)
    if not match:
        raise TaskFailure("Could not parse git repo_url")
    address_part = match.groupdict()['address_part']
    url_template = 'https://{user}:{password}@{address}'
    return url_template.format(address=address_part, user=user, password=password)


def git_init(task, repo_dir):
    with sdk2.helpers.ProcessLog(task=task, logger=logging.getLogger('git_init')) as pl:
        if not os.path.exists(repo_dir):
            os.mkdir(repo_dir)
        rc = sdk2.helpers.subprocess.Popen(['git', 'init'],
                                           cwd=repo_dir,
                                           stderr=pl.stderr,
                                           stdout=pl.stdout
                                           ).wait()
        if rc != 0:
            raise TaskFailure("git init failed")


def git_clone(task, repo_url, path, depth=0):
    logging.info('Fetching data from repository')
    clone_command = 'git clone {depth} {repo} {path}'.format(repo=repo_url,
                                                             path=path,
                                                             depth="--depth={}".format(depth) if depth else '')
    with sdk2.helpers.ProcessLog(task=task, logger=logging.getLogger('git_clone')) as pl:
        if not os.path.exists(path):
            rc = sdk2.helpers.subprocess.Popen(
                clone_command.split(),
                stderr=pl.stderr,
                stdout=pl.stdout
            ).wait()
            if rc != 0:
                raise TaskFailure("git clone failed")
        else:
            raise TaskFailure("Repo root %s does not exits" % path)


def git_fetch(task, repo_dir, remote='origin', ref_id='HEAD', depth=0, retries=2):
    fetch_command = 'git fetch {depth} {remote} {ref_id}'.format(remote=remote,
                                                                 ref_id=ref_id,
                                                                 depth="--depth={}".format(depth) if depth else '')
    with sdk2.helpers.ProcessLog(task=task, logger=logging.getLogger('git_fetch')) as pl:
        if os.path.exists(repo_dir):
            while True:
                rc = sdk2.helpers.subprocess.Popen(
                    fetch_command.split(),
                    cwd=repo_dir,
                    stderr=pl.stderr,
                    stdout=pl.stdout
                ).wait()
                if rc == 0:
                    return
                logging.info('git fetch failed, retries left: {}'.format(retries))

                if retries == 0:
                    raise TaskFailure("git fetch failed code: {}".format(rc))
                retries = retries - 1
                time.sleep(1)
        else:
            raise TaskFailure("repo directory %s does not exits" % repo_dir)


def git_checkout(task, repo_dir, ref_id):
    checkout_command = 'git checkout -f {ref_id}'.format(ref_id=ref_id)
    with sdk2.helpers.ProcessLog(task=task, logger=logging.getLogger('git_checkout')) as pl:
        if os.path.exists(repo_dir):
            rc = sdk2.helpers.subprocess.Popen(
                checkout_command.split(),
                cwd=repo_dir,
                stderr=pl.stderr,
                stdout=pl.stdout
            ).wait()
            if rc != 0:
                raise TaskFailure("git checkout failed")
        else:
            raise TaskFailure("repo directory %s does not exits" % repo_dir)
