import logging
import os

from sandbox import sdk2
from sandbox.common.errors import TaskFailure, TemporaryError
from sandbox.common.types.task import Status
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.blender.util.arcadia import svn_diff, build_directory, selective_checkout, run_tests
from sandbox.projects.blender.util.subtask import ensure_subtask_is_successful

from tempfile import NamedTemporaryFile, mkdtemp

RD_DIR = 'search/web/rearrs_upper/rearrange.dynamic'
DSSM_CONFIG_DIR = os.path.join(RD_DIR, 'blender_dssm_features')
DUMP_FMLS_DIR = 'yweb/blender/dssm_conveyor/dump_blender_fmls'
CONFIG_UPDATER_DIR = 'yweb/blender/dssm_conveyor/update_config'


def run_subprocess(cmd, logger_name, current_dir, task):
    with sdk2.helpers.ProcessLog(task, logger=logger_name) as pl:
        sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, cwd=current_dir)


class UpdateBlenderDssmModel(sdk2.Task):
    """ Delete old and add new DSSM models to RD config"""
    class Requirements(sdk2.Task.Requirements):
        disk_space = 140 * 1024  # Mb
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        do_commit = sdk2.parameters.Bool('Do commit', default_value=True)
        with do_commit.value[True]:
            ssh_key_owner = sdk2.parameters.String('Owner of the vault ssh key',
                                                   default_value='BLENDER')
            ssh_key_name = sdk2.parameters.String('Name of the vault ssh key',
                                                  default_value='robot-blendr-priemka-private')
            ssh_user_name = sdk2.parameters.String('Ssh user name',
                                                   default_value='robot-blendr-priemka')
            max_commit_retries = sdk2.parameters.Integer('Number of commit retry', default_value=3)

        run_tests = sdk2.parameters.Bool('Run tests', default_value=False)
        with run_tests.value[True]:
            test_path = sdk2.parameters.String('Tests path', required=True)
            test_timeout = sdk2.parameters.Integer('Test timeout', description='Timeout for ya tests in sec')

    class Context(sdk2.Task.Context):
        patch = None
        commit_retries = 0

    def _create_patch(self):
        arcadia_dir = selective_checkout([DUMP_FMLS_DIR, RD_DIR, CONFIG_UPDATER_DIR], self.Parameters.arcadia_url, self)
        build_directory([DUMP_FMLS_DIR, RD_DIR, CONFIG_UPDATER_DIR], arcadia_dir, self)

        # run fml_dumper
        fmls_json = NamedTemporaryFile(delete=False)
        run_subprocess([
            os.path.join(arcadia_dir, DUMP_FMLS_DIR, 'dump_blender_fmls'),
            '-p', os.path.join(arcadia_dir, RD_DIR),
            '--output', fmls_json.name
            ],
            'dump_blender_fmls', arcadia_dir, self
        )

        # run config_updater
        models_to_fmls = NamedTemporaryFile(delete=False)
        run_subprocess([
            os.path.join(arcadia_dir, CONFIG_UPDATER_DIR, 'update_config'),
            '-m', 'delete_and_add_new',
            '--fmls', fmls_json.name,
            '--config', os.path.join(arcadia_dir, DSSM_CONFIG_DIR, 'config.json'),
            '--yamake', os.path.join(arcadia_dir, DSSM_CONFIG_DIR, 'ya.make'),
            '--update_inplace',
            '--dump_models_to_fmls', models_to_fmls.name
            ],
            'config_updater', arcadia_dir, self
        )

        # make patch
        self.Context.patch = svn_diff(arcadia_dir, DSSM_CONFIG_DIR)
        logging.info('svn diff: \n{}'.format(self.Context.patch))

    def on_execute(self):

        if not self.Context.patch:
            try:
                self._create_patch()
            except Exception as e:
                raise TaskFailure(e)

        if not self.Context.patch:
            logging.info('nothing to commit')
            return

        if self.Parameters.run_tests:
            with self.memoize_stage.run_tests():
                self.Context.test_task = run_tests(
                    self.Parameters.test_path, patch=self.Context.patch, arcadia_url=self.Parameters.arcadia_url,
                    task=self
                )
                self.set_info('Wait tests')
                raise sdk2.WaitTask(self.Context.test_task, Status.Group.FINISH | Status.Group.BREAK,
                                    timeout=self.Parameters.test_timeout or None)
            ensure_subtask_is_successful(self.Context.test_task, self)

        if self.Parameters.do_commit:
            with self.memoize_stage.do_commit(commit_on_entrance=False):
                try:
                    msg = '[diff-resolver:{}] SKIP_CHECK updated blender dssm config from from: https://sandbox.yandex-team.ru/task/{}/view'.format(
                        self.author,
                        self.id
                    )
                    commit_dir = selective_checkout([DSSM_CONFIG_DIR], self.Parameters.arcadia_url, self)
                    sdk2.svn.Arcadia.apply_patch(commit_dir, self.Context.patch, mkdtemp())
                    with sdk2.ssh.Key(self, self.Parameters.ssh_key_owner, self.Parameters.ssh_key_name):
                        commit_result = sdk2.svn.Arcadia.commit(commit_dir, msg, self.Parameters.ssh_user_name)
                        if not commit_result:
                            raise TaskFailure('Empty commit result')
                        logging.info('Done commit with result: {}'.format(commit_result))
                except Exception as e:
                    if self.Context.commit_retries < self.Parameters.max_commit_retries:
                        self.Context.patch = None
                        self.Context.commit_retries += 1
                        raise TemporaryError(e)  # initiate task restart
                    else:
                        raise TaskFailure(e)
