from contextlib import contextmanager
import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia import sdk
from sandbox.sdk2 import svn


@contextmanager
def change_dir(newdir):
    prevdir = os.getcwd()
    os.chdir(os.path.expanduser(newdir))
    try:
        yield
    finally:
        os.chdir(prevdir)


def clone_arcadia(arcadia_url, task):
    tmp_dir = tempfile.mkdtemp('arcadia')
    with change_dir(tmp_dir):
        logging.info('clone arcadia to %s', tmp_dir)
        arcadia_dir = sdk.do_clone(arcadia_url, task)
    return os.path.join(tmp_dir, arcadia_dir)


def construct_logger_name(path, prefix='logger'):
    logger_name = '-'.join([os.path.basename(x) for x in path])
    return '{}_{}'.format(prefix, logger_name)[:255] # name shortened to 255 if it is too big


def selective_checkout(path, arcadia_url, task):
    arcadia_dir = clone_arcadia(arcadia_url, task)
    ya = os.path.join(arcadia_dir, 'ya')
    if not isinstance(path, list):
        path = [path]
    with sdk2.helpers.ProcessLog(task, logger=construct_logger_name(path, 'checkout')) as pl:
        sp.check_call([ya, 'make', '--checkout', '-j0'] + path, stdout=pl.stdout,
                      stderr=sp.STDOUT, cwd=arcadia_dir)
    return arcadia_dir


def build_directory(path, arcadia_dir, task):
    ya = os.path.join(arcadia_dir, 'ya')
    if not isinstance(path, list):
        path = [path]
    with sdk2.helpers.ProcessLog(task, logger=construct_logger_name(path, 'build')) as pl:
        sp.check_call([ya, 'make', '-r'] + path, stdout=pl.stdout,
                      stderr=sp.STDOUT, cwd=arcadia_dir)


def svn_export(path, arcadia_url, depth=None):
    tmp_dir = tempfile.mkdtemp(path.replace('/', '_'))
    url = svn.Arcadia.replace(
        arcadia_url, path=os.path.join(svn.Arcadia.parse_url(arcadia_url).path, path)
    )
    svn.Arcadia.export(url, tmp_dir, depth=depth)
    return tmp_dir


def svn_diff(arcadia_dir, *paths):
    ya = os.path.join(arcadia_dir, 'ya')
    paths = list(paths) if paths else ['.']
    return sp.check_output([ya, 'svn', 'diff'] + paths, cwd=arcadia_dir)


def run_tests(target, patch, arcadia_url, task):
    task = sdk2.Task['YA_MAKE'](
        task,
        description='Run tests for ' + target,
        owner=task.owner,
        checkout_arcadia_from_url=arcadia_url,
        targets=target,
        build_type='release',
        build_arch='linux',
        arcadia_patch=patch,
        test=True
    )
    return task.enqueue().id
