# -*- coding: utf-8 -*-

import os
import logging

import sandbox.common.types.client as ctc

from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import svn_last_change
from sandbox.projects.boltalka.resource_types import NLGSEARCH_CONFIG, NLGSEARCH_BINARY, NLGSEARCH_SHARD
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
import sandbox.sandboxsdk.process as process
from sandbox.sandboxsdk.task import SandboxTask


logger = logging.getLogger('BuildBoltalkaTask')


class ReleaseTypeParam(SandboxRadioParameter):
    name = 'release_type'
    description = 'Release type'
    choices = [
        ('production', 'production')
    ]
    required = True
    group = 'Release type'


class UseFuseApi(ya_make_parameters.UseArcadiaApiFuse):
    default_value = True


class BuildBoltalkaTask(nanny.ReleaseToNannyTask, YaMakeTask):
    """
        Build Boltalka binary
    """
    type = 'BUILD_BOLTALKA'

    input_parameters = [
        ya_make_parameters.ArcadiaUrl,
        ya_make_parameters.StripBinaries,
        ya_make_parameters.ArcadiaPatch,
        UseFuseApi,
        ReleaseTypeParam
    ]

    execution_space = 100 * 1024  # 40Gb will be enough
    required_ram = 1 * 1024 # 1Gb

    target_path = 'alice/boltalka/extsearch/base/nlgsearch'
    rev_attr = 'rev'

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx[consts.ALLURE_REPORT] = False
        self.ctx[consts.ALLURE_REPORT_TTL] = 4
        self.ctx['arts'] = 'alice/boltalka/extsearch/base/nlgsearch/nlgsearch'
        self.ctx['arts_source'] = 'alice/boltalka/extsearch/base/nlgsearch/search.cfg'
        self.ctx[consts.BUILD_BUNDLE_KEY] = False
        self.ctx[consts.BUILD_SYSTEM_KEY] = 'ya'
        self.ctx[consts.BUILD_TYPE_KEY] = 'release'
        self.ctx['cache_test_results'] = False
        self.ctx['check_dependencies'] = False
        self.ctx[consts.CHECK_RETURN_CODE] = True
        self.ctx['checkout'] = False
        self.ctx['checkout_mode'] = 'manual'
        self.ctx['clear_build'] = True
        self.ctx['coverage_exclude_regexp'] = '.*\.pb\.(h|cc)'
        self.ctx['disable_test_timeout'] = False
        self.ctx['force_build_depends'] = False
        self.ctx['ignore_recurses'] = False
        self.ctx['java_coverage'] = False
        self.ctx['junit_report'] = False
        self.ctx['keep_on'] = False
        self.ctx['lto'] = False
        self.ctx['make_context_on_distbuild'] = False
        self.ctx['musl'] = False
        self.ctx['pgo_add'] = False
        self.ctx['pgo_merge_timeout'] = 600
        self.ctx['privileged'] = False
        self.ctx['ram_drive_size'] = 0
        self.ctx['report_tests_only'] = False
        self.ctx['result_rd'] = 'nlgsearch basesearch binary'
        self.ctx['result_rt'] = 'NLGSEARCH_BINARY'
        self.ctx['result_single_file'] = False
        self.ctx['sandbox_tags'] = 'GENERIC'
        self.ctx['sonar'] = False
        self.ctx['sonar_default_project_filter'] = False
        self.ctx['strip_binaries'] = False
        self.ctx['targets'] = 'alice/boltalka/extsearch/base/nlgsearch;alice/boltalka/extsearch/indexer;alice/boltalka/extsearch/shard'
        self.ctx['test'] = False
        self.ctx['test_failure_code'] = 10
        self.ctx['test_log_level'] = 'debug'
        self.ctx['test_threads'] = 0
        self.ctx['tests_coverage'] = False
        self.ctx['tests_retries'] = 1
        self.ctx['thinlto'] = False
        self.ctx['use_dev_version'] = False
        self.ctx['use_system_python'] = False
        self.ctx['ya_timeout'] = 10800
        self.ctx['ya_yt_max_cache_size'] = 0
        self.ctx['ya_yt_put'] = False
        self.ctx['ya_yt_store'] = True
        logger.info('context: %s', self.ctx)

    def get_build_type(self):
        return consts.RELEASE_BUILD_TYPE

    def get_resources(self):
        return {
            NLGSEARCH_BINARY.name: {
                'resource_type': NLGSEARCH_BINARY,
                'description': 'Nlgsearch binary from BuildBoltalka',
                'resource_path': 'nlgsearch'
            },
            NLGSEARCH_CONFIG.name: {
                'resource_type': NLGSEARCH_CONFIG,
                'description': 'Nlgsearch config from BuildBoltalka',
                'resource_path': 'search.cfg'
            }
        }


    def get_resources_attrs(self):
        if self.rev_attr in self.ctx:
            release_params = self.get_release_attributes()
            return {
                NLGSEARCH_BINARY.name: release_params,
                NLGSEARCH_CONFIG.name: release_params,
                NLGSEARCH_SHARD.name: release_params,
            }
        return {}

    def on_execute(self):
        # use revision from URL or get last from SVN
        parts = self.ctx[consts.ARCADIA_URL_KEY].split('@')
        if len(parts) == 2:
            self.ctx[self.rev_attr] = parts[1]
        else:
            self.set_rev(self.get_last_rev())
        logger.info('on_execute(), context = {}'.format(self.ctx))

        return YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        logger.info('on_release(), context = {}, additional_parameters = {}'.format(self.ctx, additional_parameters))
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    def get_last_rev(self):
        if self.rev_attr in self.ctx:
            return self.ctx[self.rev_attr]
        url = '{}/{}'.format(self.ctx[consts.ARCADIA_URL_KEY], self.target_path)
        logger.info('get revision for {}'.format(url))
        return svn_last_change(url)

    def set_rev(self, rev):
        if self.rev_attr not in self.ctx:
            self.ctx[self.rev_attr] = rev
            self.ctx[consts.ARCADIA_URL_KEY] += '@' + rev

    def get_release_attributes(self, rev=None, rel=None, san=None):
        return {
            self.rev_attr: rev or self.ctx[self.rev_attr],
            'rel': rel or self.ctx[ReleaseTypeParam.name],
            'san': san or self.ctx[ya_make_parameters.Sanitize.name]
        }

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)
        token = sdk2.Vault.data('boltalka_yt_token')
        os.environ['YT_TOKEN'] = token
        process.run_process([os.path.join(output_dir, 'alice/boltalka/extsearch/shard/shard'),
                             os.path.join(source_dir, 'alice/boltalka/extsearch/shard/data.yaml'),
                             os.path.join(pack_dir, 'index'),
                             os.path.join(output_dir, 'alice/boltalka/extsearch/indexer/indexer')],
                            log_prefix='shard')
        shard = self.create_resource(
            description='Boltalka shard from BuildBoltalka',
            resource_path=os.path.join(pack_dir, 'index'),
            resource_type=NLGSEARCH_SHARD,
        )
        self.ctx['ap_packs'][NLGSEARCH_SHARD.name] = shard.id



__Task__ = BuildBoltalkaTask
