# -*- coding: utf-8 -*-
"""
* Build google-breakpad from source tarball
* Create libbreakpad_init.so to initialize breakpad via LD_PRELOAD

Author: alonger@
"""

import os
import shutil

from sandbox import sdk2
from sandbox.projects import resource_types
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers.process import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia


class BreakpadInitSoNew(sdk2.Resource):
    releasers = ["wint"]
    releasable = True


class BreakpadMinidumpStackwalkNew(sdk2.Resource):
    releasers = ["wint"]
    releasable = True


class BreakpadSymbolsDumperNew(sdk2.Resource):
    releasers = ["wint"]
    releasable = True


class BreakpadMinidumpToCoreNew(sdk2.Resource):
    releasers = ["wint"]
    releasable = True


class BuildBreakpadNew(sdk2.Task):
    """
    Builds Breakpad Init shared library
    """

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Breakpad builder'
        kill_timeout = 3600
        hidden = False
        fail_on_any_error = True
        with sdk2.parameters.Group("Resources Parameters") as resource_block:
            dependent_resource = sdk2.parameters.LastReleasedResource(
                "breakpad source code tarball",
                resource_type=resource_types.OTHER_RESOURCE,
                state=(ctr.State.READY, ctr.State.NOT_READY),
                required=True
            )

    class Context(sdk2.Task.Context):
        pass

    def res_creator(self, res_type, src_file):
        resource_out = res_type
        resource_out_data = sdk2.ResourceData(resource_out)
        resource_out_data.path.mkdir(0o755, parents=True, exist_ok=True)
        shutil.copy(os.path.abspath(src_file), str(resource_out_data.path))

    def on_execute(self):
        Arcadia.checkout(
            Arcadia.append(Arcadia.ARCADIA_TRUNK_URL, 'devtools/breakpad_init'),
            os.path.abspath('breakpad_init')
        )

        os.chdir(os.path.abspath('breakpad_init'))

        src_gz = 'breakpad.tar.gz'

        resource_src = self.Parameters.dependent_resource
        resource_data_src = sdk2.ResourceData(resource_src)
        resource_path_src = resource_data_src.path

        shutil.copy(
            str(resource_path_src),
            os.path.abspath(src_gz)
        )

        with sdk2.helpers.ProcessLog(self, logger="Unpack") as pl:
            sp.Popen(['tar', '-xvzf', src_gz], stdout=pl.stdout, stderr=sp.STDOUT).wait()
            sp.Popen(['rm', src_gz], stdout=pl.stdout, stderr=sp.STDOUT).wait()

        with sdk2.helpers.ProcessLog(self, logger="Run Build") as pl:
            sp.Popen(['chmod', '+x', 'build_breakpad_cpp11.sh'], stdout=pl.stdout, stderr=sp.STDOUT).wait()
            sp.Popen(['./build_breakpad_cpp11.sh'], stdout=pl.stdout, stderr=sp.STDOUT).wait()

        self.res_creator(
            BreakpadInitSoNew(self, self.Parameters.description, "ld_preload_so"),
            'libbreakpad_init.so'
        )
        self.res_creator(BreakpadMinidumpStackwalkNew(
            self, self.Parameters.description, "stackwalk"),
            'google-breakpad/src/processor/minidump_stackwalk'
        )
        self.res_creator(BreakpadSymbolsDumperNew(
            self, self.Parameters.description, "symbol_dumper"),
            'google-breakpad/src/tools/linux/dump_syms/dump_syms'
        )
        self.res_creator(BreakpadMinidumpToCoreNew(
            self, self.Parameters.description, "minidump-2-core"),
            'google-breakpad/src/tools/linux/md2core/minidump-2-core'
        )
