import os
import subprocess

import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc
import sandbox.projects.resource_types as rt

from sandbox import sdk2
from sandbox.common.errors import TaskFailure


class BroadmatchBuildOutput(rt.BUILD_OUTPUT):
    pass


class RunBmCommand(sdk2.Task):
    """Copy-paste from sandbox/projects/common/RunCommand with usage of latest resource"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        command = sdk2.parameters.String("Command to execute", required=True)
        env = sdk2.parameters.Dict("Environment variables. Key:Value. Use $(vault:value:owner:name) to vault items.")
        resource = sdk2.parameters.Resource("Resource to make available via {resource} in the command. Keep empty for autofill")
        client_tags = sdk2.parameters.ClientTags(
            "Client tags",
            default=ctc.Tag.GENERIC
        )

    def on_save(self):
        self.Requirements.client_tags = self.Parameters.client_tags
        if self.Parameters.resource is None:
            self.Parameters.resource = sdk2.Resource.find(type=BroadmatchBuildOutput, state=ctr.State.READY).first()

    def on_execute(self):
        command_env = os.environ.copy()

        for k, v in self.Parameters.env.items():
            if v.startswith('$(vault:value:'):
                (_, _, owner, name) = v.split(':')
                name = name[:-1]
                command_env[k] = sdk2.Vault.data(owner, name)
            else:
                command_env[k] = v

        if self.Parameters.resource:
            resource_path = str(sdk2.ResourceData(self.Parameters.resource).path)
            command = self.Parameters.command.format(resource=resource_path)
        else:
            command = self.Parameters.command

        logs_errors = []

        with sdk2.helpers.ProcessLog(self, logger="command") as pl:
            try:
                subprocess.check_call(
                    command,
                    env=command_env,
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    close_fds=True
                )
            except subprocess.CalledProcessError as e:
                logs_errors.append((pl.stderr.path, e))

        if len(logs_errors) != 0:
            last_lines = ""
            for log_err in logs_errors:
                last_lines += "== tail of {} ==\n{}\n== exception ==\n{}\n======\n".format(
                    log_err[0],
                    subprocess.check_output(["tail", "-n100", str(log_err[0])]),
                    log_err[1]
                )
            raise TaskFailure("Command failed:\n{}".format(last_lines))
