import collections
import os

from sandbox.sandboxsdk import environments, process
from sandbox import sdk2
import sandbox.projects.resource_types as rt
from sandbox.sdk2.helpers import subprocess, ProcessLog


class BroadmatchLimitationBinary(rt.ARCADIA_PROJECT):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = ["serkh"]
    release_subscribers = ["serkh"]


class BroadmatchingBinary(rt.ARCADIA_PROJECT):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = ["serkh", "firefish", "malykhin", "sashateh", "druxa"]
    release_subscribers = ["serkh"]


class BroadmatchingTask(sdk2.Task):
    @property
    def env(self):
        return {
            'YT_TOKEN': sdk2.Vault.data('robot-bm-admin', 'YT_TOKEN'),
            'YQL_TOKEN': sdk2.Vault.data('robot-bm-admin', 'YQL_TOKEN'),
        }

    def run_binary_with_args(self, args):
        """
        Run binary with list of arguments.
        self.Parameters must have 'binary_id' attribute
        Each argument is either 'scalar' (int, string etc.) or pair of scalars ('--attr-name', attr_value).

        :param args: list of arguments for binary call
        :return: None
        """

        if not hasattr(self, 'Parameters') or not hasattr(self.Parameters, 'binary_id'):
            raise AttributeError("Attribute 'binary_id' not found in self.Parameters")

        bin_res = sdk2.ResourceData(self.Parameters.binary_id)
        cmd = [str(bin_res.path)]

        for arg in args:
            if type(arg) in (list, tuple):
                if len(arg) != 2:
                    raise ValueError("Iterable arguments must have exactly 2 elements")
                cmd.append('{}={}'.format(*arg))
            else:
                cmd.append(str(arg))

        with ProcessLog(self, logger=self.__class__.__name__) as l:
            subprocess.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=self.env)


class PythonVirtualEnvironment(environments.VirtualEnvironment):
    """
    Context manager for Python virtual environment

    Usage example:

    with PythonVirtualEnvironment(pip_requirements=("yandex-yt", "yql")) as env:
        env.run_python_script([
            'some_script.py',
            '--arg', 'value'
        ], log_prefix="some_script_env")
    """

    def __init__(self, pip_requirements=None, env_vars=None):
        super(PythonVirtualEnvironment, self).__init__(use_system=True)
        self._pip_requirements = pip_requirements
        self._env_vars = env_vars

    def __enter__(self):
        self = super(PythonVirtualEnvironment, self).__enter__()
        pip_requirements = self._pip_requirements

        if pip_requirements is not None:
            if not isinstance(pip_requirements, collections.Iterable):
                raise ValueError("pip_requirements is expected to be iterable, got {}".format(pip_requirements))

            self.pip("pip setuptools==33.1.1")

            self._VirtualEnvironment__run_cmd(
                '{python} -us {pip} install -i {pypi} -vvv {req}'.format(
                    python=self.executable,
                    pip=self._VirtualEnvironment__pip_wrapper,
                    pypi='https://pypi.yandex-team.ru/simple/',
                    req=" ".join(pip_requirements),
                ), log_prefix='bm_venv'
            )
        return self

    def run_python_script(self, cmd, log_prefix='bm_venv_run_python_script'):
        """
        Run Python script and check if exit is successful.

        :param cmd: list of arguments for Python interpreter
        :param log_prefix: prefix of running command in logs
        """

        os_env = os.environ.copy()
        if self._env_vars:
            os_env.update(self._env_vars)
        process.run_process([self.executable] + cmd, environment=os_env, log_prefix=log_prefix, wait=True, check=True)


class YTVirtualEnvironment(PythonVirtualEnvironment):
    """Context manager for Python virtual environment with pre-installed YT packages"""

    def __init__(self, pip_requirements=None, env_vars=None):
        if pip_requirements is None:
            pip_requirements = set()
        else:
            pip_requirements = set(pip_requirements)

        pip_requirements.update({'yandex-yt', 'yql', 'yandex-yt-transfer-manager-client'})
        pip_requirements = list(pip_requirements)
        super(YTVirtualEnvironment, self).__init__(pip_requirements=pip_requirements, env_vars=env_vars)
