# -*- coding: utf-8 -*-

import argparse
import os

from yt.wrapper import YtClient
from yt.transfer_manager.client import TransferManager
from yql.api.v1.client import YqlClient


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--src-cluster', required=True)
    parser.add_argument('--dst-cluster', required=True)
    parser.add_argument('--src-stat-table', required=True)
    parser.add_argument('--dst-stat-table', required=True)
    parser.add_argument('--yt-pool', required=True)
    args = parser.parse_args()

    yt_token = os.environ.get('YT_TOKEN')
    yql_token = os.environ.get('YQL_TOKEN')

    yt_config = {
        'token': yt_token,
        'proxy': {'url': args.dst_cluster},
    }

    yt = YtClient(config=yt_config)

    with yt.TempTable() as tmp:
        print("Aggregate stat on {}".format(args.src_cluster))
        yql_cluster_name = args.src_cluster.replace('-', '')  # seneca-sas -> senecasas
        yql_client = YqlClient(db=yql_cluster_name, token=yql_token)
        q = """
            insert into [{tmp_dst_table}] with truncate
            from [{src_stat_table}]
            select
                OrderID, Date, IsBroadmatch, sum(Shows) as Shows, sum(Clicks) as Clicks, sum(cast(Cost as uint64)) as Cost
            where
                UpdateTime >= DateTime::ToSeconds(Yql::Now()) - 8*24*3600
            group by
                OrderID,
                DateTime::ToStringFormat(DateTime::FromSeconds(cast(UpdateTime as uint64)), '%Y-%m-%d 00:00:00') as Date,
                Broadmatch as IsBroadmatch;
        """.format(tmp_dst_table=tmp, src_stat_table=args.src_stat_table)

        request = yql_client.query(q)
        request.run()
        request.wait_progress()
        if request.is_success:
            print('YQL request OK')
        else:
            if request.errors:
                errors = [str(error) for error in request.errors]
            else:
                errors = []
            raise Exception("YQL request failed. Status: {}. Errors: {}".format(request.status, ' - '.join(errors)))

        print("Transfer from {} to {}".format(args.src_cluster, args.dst_cluster))
        tm = TransferManager(token=yt_token, retry_count=10)
        tm.add_task(args.src_cluster, tmp, args.dst_cluster, tmp, sync=True)
        yt.move(tmp, args.dst_stat_table, force=True)


if __name__ == '__main__':
    main()
