#!/usr/bin/env python

import argparse
import os
import time

from yt.wrapper import YtClient
from yt.yson import to_yson_type
from yt.transfer_manager.client import TransferManager
from yql.api.v1.client import YqlClient

DST_ROOT_DIR = '//home/serkh/broadmatch-limitation/direct-limits'
DST_DATA_DIR = DST_ROOT_DIR + '/data'
DST_STORAGE_DIR = DST_ROOT_DIR + '/storage'
DST_TABLE_SCHEMA = to_yson_type([
    {"name": "OrderHash", "sort_order": "ascending", "type": "uint64", "expression": "uint64(farm_hash(OrderID))", },
    {"name": "OrderID", "sort_order": "ascending", "type": "int64", },
    {"name": "CostLimit", "type": "int64", },
], attributes={"unique_keys": True, "strict": True, "key_columns": ["OrderHash", "OrderID"]})


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--source-cluster', required=True)
    parser.add_argument('--destination-cluster', required=True)
    parser.add_argument('--source-table', required=True)
    parser.add_argument('--destination', required=True)
    parser.add_argument('--yql-transform-query', required=True)
    parser.add_argument('--cleanup', action='store_true')
    args = parser.parse_args()

    yt_token = os.environ.get('YT_TOKEN')
    yql_token = os.environ.get('YQL_TOKEN')

    yt_config = {}
    yt_config['token'] = yt_token
    yt_config['proxy'] = {'url': args.destination_cluster}

    yt = YtClient(config=yt_config)

    print("transfer from Hahn")
    dst_table_name = args.source_table.split('/')[-1]
    dst_data_table_path = DST_DATA_DIR + '/' + dst_table_name
    if yt.exists(dst_data_table_path):
        yt.remove(dst_data_table_path)
    tm = TransferManager(token=yt_token, retry_count=10)
    tm.add_task(args.source_cluster, args.source_table, args.destination_cluster, dst_data_table_path, sync=True)

    with yt.TempTable(attributes={"schema": DST_TABLE_SCHEMA}) as tmp:
        print("load data to static table")
        yql_cluster_name = args.destination_cluster.replace('-', '')  # seneca-sas -> senecasas
        yql_client = YqlClient(db=yql_cluster_name, token=yql_token)

        queries = [
            '''
                insert into [{tmp_dst_limits_table}] with truncate
                from [{transferred_limits_table}] {yql_transform_query};
            '''.format(
                tmp_dst_limits_table=tmp,
                transferred_limits_table=dst_data_table_path,
                yql_transform_query=args.yql_transform_query
            ),
            '''
                insert into [{tmp_dst_limits_table}]
                from [//yabs/Dicts/BroadmatchLimits] as L left only join [{tmp_dst_limits_table}] as DL using(OrderID)
                select OrderID, 5 as CostLimit where L.Cost == 80;
            '''.format(tmp_dst_limits_table=tmp),
        ]
        for q in queries:
            request = yql_client.query(q)
            request.run()
            request.wait_progress()
            if request.is_success:
                print('YQL request OK')
            else:
                if request.errors:
                    errors = [str(error) for error in request.errors]
                else:
                    errors = []
                raise Exception("YQL request failed. Status: {}. Errors: {}".format(request.status, ' - '.join(errors)))

        print("convert to dynamic table")
        yt.alter_table(tmp, dynamic=True)

        print("copy to storage")
        storage_table_name = dst_table_name + '.' + str(int(time.time()))
        storage_table_path = DST_STORAGE_DIR + '/' + storage_table_name
        yt.copy(tmp, storage_table_path, force=True, recursive=True)
        yt.mount_table(storage_table_path)

        print("flip link")
        with yt.Transaction():
            if yt.exists(args.destination):
                yt.remove(args.destination)
            yt.link(storage_table_path, args.destination)

        if args.cleanup:
            print("cleanup old tables")
            for t in yt.list(DST_STORAGE_DIR):
                if t != storage_table_name:
                    yt.unmount_table(DST_STORAGE_DIR + '/' + t)
                    yt.remove(DST_STORAGE_DIR + '/' + t)
        print("all done!")


if __name__ == '__main__':
    main()
