# -*- encoding: utf-8 -*-

import datetime
import logging
import re
import textwrap

import requests

from sandbox import sdk2
from sandbox.common import errors as sc_errors
from sandbox.common import utils as sc_utils
from sandbox.sandboxsdk.environments import PipEnvironment

PLATFORM_LINUX = 'linux'
PLATFORM_MAC = 'mac'
PLATFORM_WIN = 'win'
TEAMCITY_SERVER_URL = 'https://teamcity.browser.yandex-team.ru/'
BRANDED_BUILD_TYPES_PLATFORMS = {
    'BrowserBrandedDistributives_BrandedBetaLinuxNew': PLATFORM_LINUX,
    'BrowserBrandedDistributives_BrandedBetaMac': PLATFORM_MAC,
    'BrowserBrandedDistributives_BrandedBetaWin': PLATFORM_WIN,
    'BrowserBrandedDistributives_BrandedCanaryLinuxNew': PLATFORM_LINUX,
    'BrowserBrandedDistributives_BrandedCanaryMac': PLATFORM_MAC,
    'BrowserBrandedDistributives_BrandedCanaryWin': PLATFORM_WIN,
    'BrowserBrandedDistributives_BrandedDevWin': PLATFORM_WIN,
    'BrowserBrandedDistributives_BrandedStableLinuxNew': PLATFORM_LINUX,
    'BrowserBrandedDistributives_BrandedStableMac': PLATFORM_MAC,
    'BrowserBrandedDistributives_BrandedStableWin': PLATFORM_WIN,
}
BRANDED_BUILD_BRANCHES_REGEX = re.compile('^master-(?P<version>.*)/(?P<branch_type>rc|pre)$')
ABC_INFRA_SERVICE_ID = 1769
STARTREK_QUEUE = 'BROWSER'
STARTREK_COMPONENT = 'BrandingPack'
STARTREK_PRIORITY = 'blocker'
STARTREK_TAG_BRANDED_BROKEN = 'browser-branded-broken'
STARTREK_TAG_INFRA_DUTY = 'infra-duty'
STARTREK_BRANDED_FOLLOWERS = ['edragun']
STARTREK_BRANCH_NAME = {
    'master': 'Master',
    'master-next': 'MasterNext'
}


class WatchBrowserBrandedBuildFailures(sdk2.Task):
    """
    This task processes branded build failures. It searches for issue with
    proper set of tags. If not found - creates new one with specific summary
    and description. After that task updates issue followers and posts comment
    with current infra duty as summonee.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('startrek-client==1.5.8'),
            PipEnvironment('teamcity-client==4.1.0'),
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 20 * 60  # 20 minutes.

        abc_oauth_token_vault = sdk2.parameters.String(
            'Vault item with OAuth token for ABC.',
            default='browser-infra-build-watcher-branded-builds-token')
        startrek_oauth_token_vault = sdk2.parameters.String(
            'Vault item with OAuth token for StarTrek.',
            default='browser-infra-build-watcher-branded-builds-token')
        teamcity_oauth_token_vault = sdk2.parameters.String(
            'Vault item with OAuth token for TeamCity.',
            default='browser-infra-build-watcher-branded-builds-token')
        build_id = sdk2.parameters.String('TeamCity build ID.', required=True)
        dry_run = sdk2.parameters.Bool('Dry run.', default=False)

    @sc_utils.singleton_property
    def teamcity(self):
        auth_token = sdk2.Vault.data(
            self.Parameters.teamcity_oauth_token_vault)
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=TEAMCITY_SERVER_URL, auth=auth_token)

    @sc_utils.singleton_property
    def startrek(self):
        auth_token = sdk2.Vault.data(
            self.Parameters.startrek_oauth_token_vault)
        import startrek_client
        return startrek_client.Startrek(
            token=auth_token, useragent='sandbox-task')

    @staticmethod
    def get_nearest_duty(date, abc_token):
        """
        If there is duty on `date`, return it. If not, return nearest duty in the future.
        """
        response = requests.get(
            'https://abc-back.yandex-team.ru/api/v4/duty/shifts/',
            params={
                'service': ABC_INFRA_SERVICE_ID,
                'date_from': date.strftime('%Y-%m-%d'),
                'date_to': (date + datetime.timedelta(days=7)).strftime('%Y-%m-%d'),
                'fields': 'is_approved,person.login,replaces',
            },
            headers={'Authorization': 'OAuth {}'.format(abc_token)})
        response.raise_for_status()
        for shift in response.json().get('results', []):
            if not shift['is_approved']:
                continue
            for replace in shift.get('replaces', []):
                if not replace['is_approved']:
                    continue
                start = datetime.datetime.strptime(replace['start'], '%Y-%m-%d').date()
                end = datetime.datetime.strptime(replace['end'], '%Y-%m-%d').date()
                if start <= date <= end:
                    return replace['person']['login']
            return shift['person']['login']
        raise RuntimeError('Failed to determine duty')

    @staticmethod
    def create_query(platform, version):
        conditions = [
            'Resolution: empty()',
            'Tags: "{}"'.format(STARTREK_TAG_BRANDED_BROKEN),
            'Tags: "{}"'.format(platform),
            '"Affected Version": "{}"'.format(version)
        ]
        return ' '.join(conditions)

    def create_issue(self, dry_run, platform, version, summary, description):
        tags = [STARTREK_TAG_BRANDED_BROKEN, STARTREK_TAG_INFRA_DUTY, platform]
        if dry_run:
            logging.info('[DRY] Create issue:')
            logging.info('[DRY] - queue: %s', STARTREK_QUEUE)
            logging.info('[DRY] - summary: %s', summary)
            logging.info('[DRY] - tags: %s', tags)
            logging.info('[DRY] - version: %s', version)
            logging.info('[DRY] - description:\n%s', description)
            return None
        else:
            issue = self.startrek.issues.create(
                queue=STARTREK_QUEUE,
                summary=summary,
                description=description,
                tags=tags,
                affectedVersions=[version],
                fixVersions=[version])
            logging.info('Issue created: %s', issue.key)
            return issue

    @staticmethod
    def get_summary(platform, version):
        return ('Поломка branded-сборки. '
                'Платформа {platform}, версия {version}').format(
            platform=platform, version=version)

    @staticmethod
    def get_description():
        return textwrap.dedent('''
            В одной из branded-сборок случилась поломка.
            Ссылка на сломанную сборку находится в комментариях.
            Необходимо проверить branded-сборки для всех платформ в этой ветке.
            Если падение разовое из-за мигающей проблемы компиляции, то
            нужно изменить приоритет на "Критичный" и забрать задачу в BYIN.
        ''')

    @staticmethod
    def get_comment(build_url, platform, version):
        return ('Branded-сборка (платформа {platform}, версия {version}) '
                '(({build_url} завершилась с ошибкой)).').format(
            build_url=build_url, platform=platform, version=version)

    def on_execute(self):
        abc_token = sdk2.Vault.data(self.Parameters.abc_oauth_token_vault)
        tc_build_id = self.Parameters.build_id
        dry_run = self.Parameters.dry_run

        # Get TeamCity build and check status.

        build = self.teamcity.Build(id=tc_build_id)
        if build.status != 'FAILURE':
            raise sc_errors.TaskError(
                'Build status is not FAILURE, but: {}'.format(build.status))

        # Get duty.
        duty = self.get_nearest_duty(datetime.date.today(), abc_token)
        logging.info('Duty is: %s', duty)

        # Get platform and version.

        platform = BRANDED_BUILD_TYPES_PLATFORMS.get(build.build_type.id)
        if not platform:
            raise sc_errors.TaskError(
                'Wrong build type: {}'.format(build.build_type.id))
        logging.info('platform : %s', platform)

        if build.branch_name in ('master', 'master-next'):
            version = STARTREK_BRANCH_NAME[build.branch_name]
        else:
            branch_matcher = BRANDED_BUILD_BRANCHES_REGEX.match(build.branch_name)
            if not branch_matcher:
                raise sc_errors.TaskError(
                    'Non-release branch: {}'.format(build.branch_name))
            else:
                version = branch_matcher.group('version')
                if branch_matcher.group('branch_type') == 'pre':
                    version += '/pre'
        logging.info('version : %s', version)

        # Find or create issue.

        query = self.create_query(platform, version)
        issues = self.startrek.issues.find(query=query)
        if issues:
            issue = issues[0]
            logging.info('Issue found: %s', issue.key)
        else:
            st_summary = self.get_summary(platform, version)
            st_description = self.get_description()
            issue = self.create_issue(
                dry_run, platform, version, st_summary, st_description)

        # Prepare dict to update issue.

        issue_update_dict = {}

        # Assignee.
        if issue and not issue.assignee:
            if dry_run:
                logging.info('[DRY] Set assignee: %s', duty)
            else:
                issue_update_dict['assignee'] = duty
                logging.info('Set assignee: %s', duty)

        # Component. Only for default queue.
        components = [comp.name for comp in issue.components] if issue else []
        logging.info('Issue components: %s', components)
        if issue and issue.queue.key == STARTREK_QUEUE and STARTREK_COMPONENT not in components:
            if dry_run:
                logging.info('[DRY] Append component: %s', STARTREK_COMPONENT)
            else:
                issue_update_dict['components'] = components + [STARTREK_COMPONENT]
                logging.info('Component appended: %s', STARTREK_COMPONENT)

        # Followers.
        followers = [user.id for user in issue.followers] if issue else []
        logging.info('Issue followers: %s', followers)
        if any(person not in followers for person in STARTREK_BRANDED_FOLLOWERS):
            if dry_run:
                logging.info('[DRY] Append followers: %s', STARTREK_BRANDED_FOLLOWERS)
            else:
                issue_update_dict['followers'] = followers + STARTREK_BRANDED_FOLLOWERS
                logging.info('Followers appended: %s', STARTREK_BRANDED_FOLLOWERS)

        # Task status.
        if issue and issue.priority.key != STARTREK_PRIORITY:
            if dry_run:
                logging.info('[DRY] Update task priority: %s', STARTREK_PRIORITY)
            else:
                issue_update_dict['priority'] = STARTREK_PRIORITY
                logging.info('Update task priority: %s', STARTREK_PRIORITY)

        # Update issue.

        if issue_update_dict:
            issue.update(**issue_update_dict)

        # Add branded build failed comment.
        st_comment = self.get_comment(build.web_url, platform, version)
        if dry_run:
            logging.info('[DRY] Add comment: %s', st_comment)
        else:
            issue.comments.create(text=st_comment, summonees=[duty])
            logging.info('Comment created')
