# -*- encoding: utf-8 -*-
import logging

from sandbox import common
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.browser.ab_experiments import ab_service
from sandbox.projects.browser.ab_experiments import base
from sandbox.projects.browser.ab_experiments import channels
from sandbox.projects.browser.ab_experiments.BrowserExperimentsAbChecklist import checklists


class BrowserExperimentsAbChecklistTask(base.BaseAbTask):
    _TARGET_AB_TASK_TYPES = (ab_service.ABT_TYPE, ab_service.FEATURE_TOGGLE_TYPE, )

    class Requirements(base.AbTaskRequirements):
        environments = [
            PipEnvironment('startrek_client', version='2.5',
                           custom_parameters=['--upgrade-strategy only-if-needed']),
        ]

    @common.utils.singleton_property
    def startrek_client(self):
        import startrek_client
        return startrek_client.Startrek(useragent=self.__class__.name,
                                        token=self.oauth_token,
                                        headers={'Accept-Language': 'en-US, en'})

    def _on_execute(self):
        super(BrowserExperimentsAbChecklistTask, self)._on_execute()

        network = self.ab_task_info['restrictions'].get('networks')
        platforms = self.ab_task_info['restrictions'].get('browser__platforms')
        platforms = platforms.split(',') if platforms else ['']
        ab_channels = self.ab_task_info['restrictions'].get('browser__build_types')
        ab_channels = ab_channels.split(',') if ab_channels else []

        exp_channels = set(filter(
            None,
            map(channels.get_channel_by_ab_channel, ab_channels)
        ))
        exp_channels = sorted(exp_channels, key=lambda ch: ch.value)

        logging.info('type: %s, AB channels: %s, exp channels: %s, ntework: %s, platforms: %s',
                     self.ab_task_type, ab_channels, exp_channels, network, platforms)

        issue = self.startrek_client.issues[self.ab_task_id]

        current_checklist_item_ids = [item['id'] for item in issue.checklistItems]
        current_checklist = [
            checklists.ChecklistItem(item['text'], item['checked'])
            for item in issue.checklistItems
        ]

        exp_info = []
        for channel in exp_channels:
            for platform in platforms:
                exp_info.append((self.ab_task_type, channel, platform,  network))

        new_checklist = checklists.build_actual_checklist(current_checklist, self.ab_task_queue, exp_info)
        logging.info('New checklist for issue %s has %d items', issue.key, len(new_checklist))

        if new_checklist == current_checklist:
            logging.info('Issue %s already has an actual checklist', issue.key)
            return

        # We do not delete all existing items at once before adding new checklist
        # because we will lost information about checked items in case of adding checklist request failed
        self._add_checklist(issue, new_checklist)
        for item_id in current_checklist_item_ids:
            self._try_delete_checklist_item(issue, item_id)

    def _add_checklist(self, issue, checklist):
        """
        :type issue: startrek_client.objects.Resource
        :type checklist: list[checklists.ChecklistItem]
        """
        logging.info('Adding to the issue %s checklist of %d items', issue.key, len(checklist))

        if not checklist:
            logging.info('Skip adding empty checklist to %s', issue.key)
            return

        data = [{'text': item.text, 'checked': item.checked} for item in checklist]
        if self.Parameters.dry_run:
            logging.info('Skip adding checklist to issue %s because of dry run mode', issue.key)
            return
        issue.perform_action('checklistItems', 'post', list_data=data)

    def _try_delete_checklist_item(self, issue, checklist_item_id):
        """
        :type issue: startrek_client.objects.Resource
        :type checklist_item_id: str
        """
        from startrek_client import exceptions

        logging.info('Deleting checklist item "%s" from issue %s', checklist_item_id, issue.key)
        if self.Parameters.dry_run:
            logging.info('Skip deleting checklist item "%s" from issue %s because of dry run mode',
                         checklist_item_id, issue.key)
            return
        try:
            issue.perform_action('checklistItems/{}'.format(checklist_item_id), 'delete',
                                 params={'notify': False},
                                 ignore_empty_body=True)
        except exceptions.UnprocessableEntity as exc:
            logging.info('Checklist item "%s" from issue %s does not exist: %s',
                         checklist_item_id, issue.key, exc)
