from collections import defaultdict

import enum


class ChecklistsMergeError(RuntimeError):
    pass


def merge_checklists(checklists, checked_items, ignore_cycles=False):
    """Kinda topological sort

    :param checklists: checklists to be merged
    :param checked_items: set of checked items, used for prioritizing items
    :param ignore_cycles: if `True` break possible cyclic dependencies between items

    :type checklists: list[list[str]]
    :type checked_items: set[str]
    :type ignore_cycles: bool

    :rtype: list[str]
    """
    if not checklists:
        return []

    def nonchecked_items_weight(checklist):
        # the closer an item is to the beginning of the checklist,
        # the greater the item's contribution to the total sum
        return sum(
            2 ** (-0.2 * item_pos)
            for item_pos, item in enumerate(checklist) if item not in checked_items
        )

    # order checklists by the weight of non-checked items
    checklists.sort(key=nonchecked_items_weight)

    item_first_checklist = {}
    for checklist_id, checklist in enumerate(checklists):
        for item in checklist:
            item_first_checklist.setdefault(item, checklist_id)

    uniq_items = set.union(*map(set, checklists))
    items = sorted(uniq_items, key=lambda item: (item_first_checklist[item], item), reverse=True)
    items_ids = {item: id_ for id_, item in enumerate(items)}

    edges = defaultdict(set)
    for checklist in checklists:
        for item_pos in range(len(checklist) - 1):
            from_ = items_ids[checklist[item_pos]]
            to = items_ids[checklist[item_pos + 1]]
            edges[from_].add(to)

    class DfsStates(enum.Enum):
        UNVISITED = 0  # means we haven't considered this item yet
        VISITED = 1  # means we've already visited this item, but haven't added it to the resulting list yet
        PICKED = 2  # means we've already added this item to the resulting checklist

    item_states = [DfsStates.UNVISITED] * len(items)

    merged_checklist_items = []

    def dfs(cur_item_id):
        item_states[cur_item_id] = DfsStates.VISITED
        for next_item_id in sorted(edges[cur_item_id]):
            if item_states[next_item_id] == DfsStates.VISITED and not ignore_cycles:
                raise ChecklistsMergeError(
                    u'Checklists are inconsistent: cycle at the vertex "{}"'.format(items[next_item_id]))
            if item_states[next_item_id] != DfsStates.UNVISITED:
                continue
            dfs(next_item_id)
        item_states[cur_item_id] = DfsStates.PICKED
        merged_checklist_items.append(cur_item_id)

    for item_id in sorted(items_ids.values()):
        if item_states[item_id] == DfsStates.UNVISITED:
            dfs(item_id)

    merged_items = [items[item_id] for item_id in reversed(merged_checklist_items)]
    assert len(merged_items) == len(uniq_items)

    return merged_items
