import logging

import enum
import requests
from six.moves import urllib


@enum.unique
class Queue(enum.Enum):
    # IDs from AB service
    GATEWAY = 8
    DEBRO = 37
    BROWSER = 67
    IBRO = 86


_AB_API_BASE_URL = 'https://ab.yandex-team.ru/api/'
ABT_TYPE = 'ABT'
FEATURE_TOGGLE_TYPE = 'FEATURE_TOGGLE'


def _bind_request_method(http_method):
    def method(self, url, *args, **kwargs):
        return self._request(http_method, url, *args, **kwargs)

    method.func_name = http_method.lower()
    return method


class ABClient(object):
    def __init__(self, useragent, token=None, base_url=None):
        self._base_url = base_url or _AB_API_BASE_URL
        self._session = requests.Session()
        self._session.headers['User-Agent'] = useragent
        if token:
            self._session.headers['Authorization'] = 'OAuth {}'.format(token)
        retry = requests.packages.urllib3.Retry(
            total=10, backoff_factor=0.5,
            status_forcelist=frozenset([500, 502, 503, 504]),
        )
        self._session.mount(self._base_url, requests.adapters.HTTPAdapter(max_retries=retry))

    @staticmethod
    def _make_url(url_template, *args):
        return url_template.format(*(urllib.parse.quote(s, safe='') for s in args))

    get = _bind_request_method('GET')
    delete = _bind_request_method('DELETE')
    put = _bind_request_method('PUT')
    post = _bind_request_method('POST')

    def _request(self, method, url, params=None):
        url = urllib.parse.urljoin(self._base_url, url)
        response = self._session.request(method, url, params=params)
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError:
            logging.error('%s %s: %s (%s)',
                          method.upper(), url, response.status_code, response.text)
            raise
        if not response.content:
            return None
        return response.json()

    def get_task(self, task_id):
        url = self._make_url('task/{}', task_id)
        return self.get(url)
