import logging

from sandbox import common
import sandbox.common.types.client as ctc
from sandbox.common.types import misc as ctm
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox import sdk2

from sandbox.projects.browser.ab_experiments import ab_service


ROBOT_BRO_EXP_SECRET_ID = 'sec-01e12487jzcbbkt0644wf53mbj'


class AbTaskRequirements(sdk2.Task.Requirements):
    client_tags = ctc.Tag.BROWSER
    dns = ctm.DnsType.DNS64
    cores = 1

    class Caches(sdk2.Requirements.Caches):
        pass


class AbTaskParameters(sdk2.Parameters):
    kill_timeout = 5 * 60  # 5 min

    task_id = sdk2.parameters.StrictString('Task ID', regexp=r'^[A-Z]+-\d+$', required=True)
    dry_run = sdk2.parameters.Bool('Dry run')

    with sdk2.parameters.Group('Credentials') as credentials:
        oauth_token_secret = sdk2.parameters.YavSecret(
            'OAuth token secret',
            default=sdk2.yav.Secret(ROBOT_BRO_EXP_SECRET_ID, None, 'oauth_token'),
        )


class BaseAbTask(sdk2.Task):
    _ALLOWED_AB_QUEUES = ab_service.Queue
    _TARGET_AB_TASK_TYPES = (ab_service.ABT_TYPE, )

    class _BadAbQueueError(ValueError):
        pass

    class _BadAbTaskTypeError(ValueError):
        pass

    class Requirements(AbTaskRequirements):
        pass

    class Parameters(AbTaskParameters):
        pass

    def on_enqueue(self):
        super(BaseAbTask, self).on_enqueue()
        if not self.Parameters.dry_run:
            self.Parameters.notifications += [
                sdk2.Notification(
                    statuses=[
                        ctt.Status.EXCEPTION,
                        ctt.Status.TIMEOUT,
                    ],
                    recipients=['browser-infra-bots'],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

    @property
    def oauth_token(self):
        if not self.Parameters.oauth_token_secret.default_key:
            raise ValueError('Specify yav secret key for oauth token')
        return self.Parameters.oauth_token_secret.value()

    @common.utils.singleton_property
    def ab_client(self):
        return ab_service.ABClient(useragent=self.__class__.__name__,
                                   token=self.oauth_token)

    @property
    def ab_task_id(self):
        return self.Parameters.task_id

    @common.utils.singleton_property
    def ab_task_info(self):
        return self.ab_client.get_task(self.ab_task_id)

    @common.utils.singleton_property
    def ab_task_queue(self):
        """
        :rtype: ab_service.Queue
        :raises: common.errors.TaskFailure
        """
        queue_id = self.ab_task_info['queue_id']
        try:
            queue = ab_service.Queue(queue_id)
        except ValueError as exc:
            raise self._BadAbQueueError(str(exc))
        if queue not in self._ALLOWED_AB_QUEUES:
            msg = 'Queue {} (id {}) is not a valid choice'.format(queue, queue_id)
            raise self._BadAbQueueError(msg)
        return queue

    @common.utils.singleton_property
    def ab_task_type(self):
        """
        :rtype: str
        :raises: common.errors.TaskFailure
        """
        task_type = self.ab_task_info['type']
        if task_type not in self._TARGET_AB_TASK_TYPES:
            msg = 'Skip processing task {} as its type "{}" is not in the set of target types {}'.format(
                self.ab_task_id, task_type, self._TARGET_AB_TASK_TYPES)
            raise self._BadAbTaskTypeError(msg)
        return task_type

    def on_execute(self):
        try:
            # Validating queue id
            queue = self.ab_task_queue
        except self._BadAbQueueError as exc:
            raise common.errors.TaskFailure(str(exc))
        logging.info('Task %s, AB queue: %s', self.ab_task_id, queue)
        try:
            self.ab_task_type
        except self._BadAbTaskTypeError as exc:
            logging.info(str(exc))
            return
        self._on_execute()

    def _on_execute(self):
        pass
